C-----------------------------------------------------------------------
      SUBROUTINE GNI_IRK2(N,FCN,NSTEP,X,P,Q,XEND,
     &                    METH,SOLFIX,IOUT,RPAR,IPAR)
C-----------------------------------------------------------------------
C                 VERSION OF SEPTEMBER 4,2002  
C  E-MAIL CONTACT ADDRESS : Ernst.Hairer@math.unige.ch
C-----------------------------------------------------------------------
C  SOLVES SECOND ORDER ORDINARY DIFFERENTIAL EQUATIONS OF THE FORM
C                       Q'' = F(X,Q)
C  BASED ON THE SYMPLECTIC AND SYMMETRIC GAUSS (IRK) METHODS
C  DESCRIBED IN SECTIONS II.1, VIII.6 OF THE BOOK:
C
C      E. HAIRER, C. LUBICH, G. WANNER, GEOMETRIC NUMERICAL INTEGRATION,
C         STRUCTURE-PRESERVING ALGORITHMS FOR ODES.
C         SPRINGER SERIES IN COMPUT. MATH. 31, SPRINGER 2002.
C
C  AND IN THE PUBLICATION
C
C      E. HAIRER, M. HAIRER, GNI-CODES - MATLAB PROGRAMS FOR
C         GEOMETRIC NUMERICAL INTEGRATION.
C
C  INPUT..
C     N           DIMENSION OF Q AND F(X,Q) 
C
C     FCN         NAME (EXTERNAL) OF SUBROUTINE COMPUTING F(X,Q):
C                    SUBROUTINE FCN(N,X,Q,F,RPAR,IPAR)
C                    REAL*8 Q(N),F(N)
C                    F(1)=...   ETC.
C
C     NSTEP       NUMBER OF INTEGRATION STEPS
C                    CONSTANT STEP SIZE, H=(XEND-X)/NSTEP
C
C     X           INITIAL X-VALUE
C     P(N)        INITIAL VELOCITY VECTOR
C     Q(N)        INITIAL POSITION VECTOR
C     XEND        FINAL X-VALUE
C
C     METH        NUMBER OF STAGES OF THE GAUSS METHOD
C                    FOR THE MOMENT ONLY POSSIBLE VALUES: 2,4,6.
C
C     SOLFIX      NAME (EXTERNAL) OF SUBROUTINE PROVIDING THE
C                 NUMERICAL SOLUTION DURING INTEGRATION. 
C                 IF IOUT=1, IT IS CALLED AFTER EVERY STEP.
C                 SUPPLY A DUMMY SUBROUTINE IF IOUT=0. 
C                    SUBROUTINE SOLFIX (NR,XOLD,X,P,Q,N,IRTRN,RPAR,IPAR)
C                    DOUBLE PRECISION X,Y(N),CONT(LRC)
C                      ....  
C                 SOLFIX FURNISHES THE SOLUTION "Q,P" AT THE NR-TH
C                    GRID-POINT "X" (INITIAL VALUE FOR NR=0).
C                 "XOLD" IS THE PRECEEDING GRID-POINT.
C                 "IRTRN" SERVES TO INTERRUPT THE INTEGRATION. IF IRTRN
C                    IS SET <0, RETURN TO THE CALLING PROGRAM.
C     IOUT        SWITCH FOR CALLING THE SUBROUTINE SOLFIX:
C                    IOUT=0: SUBROUTINE IS NEVER CALLED
C                    IOUT=1: SUBROUTINE IS AVAILABLE FOR OUTPUT.
C
C     RPAR(LR)    REAL PARAMETER ARRAY; LR MUST BE AT LEAST LR=10
C                    RPAR(1),...,RPAR(10) SERVE AS PARAMETERS FOR
C                    THE CODE. FURTHER VALUES CAN BE USED FOR DEFINING
C                    PARAMETERS IN THE PROBLEM
C     IPAR(LI)    INTEGER PARAMETER ARRAY; LI MUST BE AT LEAST LI=10
C                    IPAR(1),...,IPAR(10) SERVE AS PARAMETERS FOR
C                    THE CODE. FURTHER VALUES CAN BE USED FOR DEFINING
C                    PARAMETERS IN THE PROBLEM
C
C  OUTPUT..
C     P(N)        SOLUTION (VELOCITY) AT XEND
C     Q(N)        SOLUTION (POSITION) AT XEND
C-----------------------------------------------------------------------
C     SOPHISTICATED SETTING OF PARAMETERS 
C-----------------------------------------------------------------------
C    RPAR(1)   UROUND, THE ROUNDING UNIT, DEFAULT 1.D-16.
C    IPAR(1)   NITMAX, MAXIMAL NUMER OF FIXED POINT ITERAT., DEFAULT 50
C-----------------------------------------------------------------------
      PARAMETER (NDGL=500,NSD=6,NMD=3)
      IMPLICIT REAL*8 (A-H,O-Z)
      DIMENSION F(NDGL*NSD),EQ(NDGL),EP(NDGL),YH(NDGL),QQ(NDGL)
      DIMENSION C(NSD),AA(NSD,NSD),E(NSD,NSD+NMD),B(NSD),BC(NSD)
      DIMENSION SM(NMD),AMF(NMD,NSD+NMD),AM(NSD+NMD)
      DIMENSION Q(N),P(N),FS(NDGL),PS(NDGL),ZQ(NDGL,NSD)
      DIMENSION IPAR(*),RPAR(*)
      EXTERNAL FCN
C -------- UROUND   SMALLEST NUMBER SATISFYING 1.0D0+UROUND>1.0D0  
      IF (RPAR(1).EQ.0.0D0) THEN
         UROUND=1.0D-16
      ELSE
         UROUND=RPAR(1)
      END IF
C -------- NITMAX, MAXIMAL NUMER OF FIXED POINT ITERATIONS
      IF (IPAR(1).EQ.0) THEN
         NITMAX=50
      ELSE
         NITMAX=IPAR(1)
      END IF
C --------
      NS=METH
      H=(XEND-X)/NSTEP
      CALL COEFG(NS,C,B,BC,NSD,AA,E,NM,SM,NMD,AM,H)
      IF (IOUT.NE.0) CALL SOLFIX (0,X,X,P,Q,N,IRTRN,RPAR,IPAR)
      CALL FCN(N,X,Q,FS,RPAR,IPAR)
      DO IS=1,NS
        FAC=C(IS)**2/2
        DO I=1,N
          ZQ(I,IS)=C(IS)*P(I)+FAC*FS(I)
        END DO
      END DO
      DO I=1,N
        PS(I)=P(I)
        EQ(I)=0.0D0
        EP(I)=0.0D0
      END DO
C --- LOOP FOR THE ITERATIONS
      DO ISTEP=1,NSTEP
        IF (ISTEP.GT.1) CALL STARTB (FCN,N,X,P,Q,NS,NDGL,FS,PS,
     &                   ZQ,NSD,E,YH,NM,SM,AM,F,C,RPAR,IPAR)
C --- FIXED POINT ITERATION
        NITER=0
        DYNOLD=0.0D0
  40    CONTINUE
        CALL RKNITE(FCN,N,NS,X,Q,P,NSD,AA,C,NDGL,QQ,ZQ,F,DYNO,RPAR,IPAR)
        NITER=NITER+1
        IF (DYNOLD.LT.DYNO.AND.DYNO.LT.10*UROUND) GOTO 50
        IF (NITER.GE.NITMAX) THEN
          WRITE (6,*) 'NO CONVERGENCE OF ITERATION',DYNO
          IF (DYNO.GT.UROUND*1.D6) RETURN
        END IF
        IF (DYNO.GT.0.1D0*UROUND) GOTO 40
 50     CONTINUE
C --- UPDATE OF THE SOLUTION
        X=X+H
        DO I=1,N
          EQI=EQ(I)
          QI=Q(I)
          AY=QI
          EQI=EQI+H*P(I)
          QI=AY+EQI
          EQI=EQI+(AY-QI)
          DO IS=1,NS
            AY=QI
            EQI=EQI+F(I+(IS-1)*N)*BC(IS)
            QI=AY+EQI
            EQI=EQI+(AY-QI)
          END DO
          AY=Q(I)
          Q(I)=QI
          EQ(I)=EQI
          EPI=EP(I)
          PI=P(I)
          DO IS=1,NS/2
            AY=PI
            EPI=EPI+(F(I+(IS-1)*N)+F(I+(NS-IS)*N))*B(IS)
            PI=AY+EPI
            EPI=EPI+(AY-PI)
          END DO
          P(I)=PI
          EP(I)=EPI
        END DO
        IF (IOUT.NE.0) CALL SOLFIX (ISTEP,X-H,X,P,Q,N,IRTRN,RPAR,IPAR)
      END DO
      RETURN
      END
C
      SUBROUTINE STARTB (FCN,N,X,P,Q,NS,NDGL,FS,PS,ZQ,NSD,E,YH,
     &                   NM,SM,AM,F,C,RPAR,IPAR)
C ----------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
      REAL*8 ZQ(NDGL,NS),E(NSD,NS+NM),PS(N),F(N*NS),C(NS)
      REAL*8 AM(NS+NM),SM(NM),P(N),Q(N),YH(NDGL),FS(NDGL)
      DIMENSION IPAR(*),RPAR(*)
        NS1=NS+1
        NS2=NS+2
        NSM=NS+NM
        DO I=1,N
          SAV=0.0D0
          DO JS=1,NS
            SAV=SAV+AM(JS)*ZQ(I,JS)
          END DO
          YH(I)=SAV+AM(NS1)*PS(I)+AM(NS2)*P(I)+Q(I)
          DO IS=1,NS
            SAV=0.0D0
            DO JS=1,NS
              SAV=SAV+E(IS,JS)*F(I+(JS-1)*N)
            END DO
            ZQ(I,IS)=SAV+E(IS,NS1)*FS(I)
          END DO
        END DO
C
        CALL FCN(N,X+H,Q,FS,RPAR,IPAR)
        CALL FCN(N,X+H*SM(NM),YH,F,RPAR,IPAR)
        DO I=1,N
          PS(I)=P(I)
          DO IS=1,NS
            ZQIIS=ZQ(I,IS)+E(IS,NS2)*FS(I)+E(IS,NSM)*F(I)
            ZQ(I,IS)=ZQIIS+C(IS)*P(I)
          END DO
        END DO
      RETURN
      END
C
      SUBROUTINE RKNITE(FCN,N,NS,X,Q,P,NSD,AA,C,NDGL,QQ,ZQ,
     &                  F,DYNO,RPAR,IPAR)
C ----------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
      REAL*8 ZQ(NDGL,NS),C(NS),Q(N),P(N)
      REAL*8 AA(NSD,NS),F(NDGL*NS),QQ(NDGL)
      DIMENSION IPAR(*),RPAR(*)
      EXTERNAL FCN
C ---
      DO JS=1,NS
        DO J=1,N
          QQ(J)=Q(J)+ZQ(J,JS)
        END DO
        CALL FCN(N,X+C(JS),QQ,F(1+(JS-1)*N),RPAR,IPAR)
      END DO
C ---
      DYNO=0.D0
      DO I=1,N
        DNOM=MAX(1.0D-1,ABS(Q(I)))
        DO IS=1,NS
          SUM=C(IS)*P(I)
          DO JS=1,NS
            SUM=SUM+AA(IS,JS)*F(I+(JS-1)*N)
          END DO
          DYNO=DYNO+((SUM-ZQ(I,IS))/DNOM)**2
          ZQ(I,IS)=SUM
        END DO
      END DO
      DYNO=DSQRT(DYNO/(NS*N))
C ---
      RETURN
      END
C
      SUBROUTINE COEFG(NS,C,B,BC,NSD,AA,E,NM,SM,NMD,AM,HSTEP)
      IMPLICIT REAL*8 (A-H,O-Z)
      REAL*8 C(NS),AA(NSD,NS),E(NSD,NS+NM),B(NS)
      REAL*8 AM(NS+NM),SM(NM),BC(NS)
      NM=3
      IF (NS.EQ.2) THEN
         C(1)= 0.21132486540518711775D+00
         C(2)= 0.78867513459481288225D+00
         B(1)= 0.50000000000000000000D+00
         B(2)= 0.50000000000000000000D+00
         BC(1)= 0.39433756729740644113D+00
         BC(2)= 0.10566243270259355887D+00
         AA(1,1)= 0.41666666666666666667D-01
         AA(1,2)=-0.19337567297406441127D-01
         AA(2,1)= 0.26933756729740644113D+00
         AA(2,2)= 0.41666666666666666667D-01
         E(1,1)=-0.28457905077110526160D-02
         E(1,2)=-0.63850024471784160410D-01
         E(1,3)= 0.48526095198694517563D-02
         E(1,4)= 0.11305688530429939012D+00
         E(1,5)=-0.28884580475413403312D-01
         E(2,1)= 0.41122751744511433137D-01
         E(2,2)=-0.18654814888622834132D+00
         E(2,3)=-0.18110185277445209332D-01
         E(2,4)= 0.36674109449368040786D+00
         E(2,5)= 0.10779872188955481745D+00
         SM(1)= 0.00000000000000000000D+00
         SM(2)= 0.10000000000000000000D+01
         SM(3)= 0.16000000000000000000D+01
         AM(1)= 0.25279583039343438291D+02
         AM(2)=-0.86907830393434382912D+01
         AM(3)=-0.80640000000000000000D+00
         AM(4)= 0.29184000000000000000D+01
         AM(5)= 0.00000000000000000000D+00
      END IF
      IF (NS.EQ.4) THEN
         C(1)= 0.69431844202973712388D-01
         C(2)= 0.33000947820757186760D+00
         C(3)= 0.66999052179242813240D+00
         C(4)= 0.93056815579702628761D+00
         B(1)= 0.17392742256872692869D+00
         B(2)= 0.32607257743127307131D+00
         B(3)= 0.32607257743127307131D+00
         B(4)= 0.17392742256872692869D+00
         BC(1)= 0.16185132086231030665D+00
         BC(2)= 0.21846553629538057030D+00
         BC(3)= 0.10760704113589250101D+00
         BC(4)= 0.12076101706416622036D-01
         AA(1,1)= 0.40381914508467311298D-02
         AA(1,2)=-0.32958609449446961650D-02
         AA(1,3)= 0.26447829520668538006D-02
         AA(1,4)=-0.97672296325588161023D-03
         AA(2,1)= 0.43563580902396261254D-01
         AA(2,2)= 0.13818951406296126013D-01
         AA(2,3)=-0.43401341944349953440D-02
         AA(2,4)= 0.14107297391595337720D-02
         AA(3,1)= 0.10586435263357640763D+00
         AA(3,2)= 0.10651836096505307395D+00
         AA(3,3)= 0.13818951406296126013D-01
         AA(3,4)=-0.17580153590805494993D-02
         AA(4,1)= 0.14879849619263780300D+00
         AA(4,2)= 0.19847049885237718995D+00
         AA(4,3)= 0.81671359795877570687D-01
         AA(4,4)= 0.40381914508467311298D-02
         E(1,1)=-0.21272768296134340207D-01
         E(1,2)= 0.11059138674756969912D-01
         E(1,3)= 0.38999255049973564023D-02
         E(1,4)=-0.43986226789008967612D-01
         E(1,5)= 0.13581590305438849621D-01
         E(1,6)= 0.39922421675314269059D-01
         E(1,7)=-0.79369058065113002021D-03
         E(2,1)=-0.75671119283734809953D-02
         E(2,2)= 0.10209394000843457002D-01
         E(2,3)=-0.12880197817980892596D-01
         E(2,4)=-0.56381316813776501277D-01
         E(2,5)= 0.37440782682669799960D-02
         E(2,6)= 0.11522469441011273193D+00
         E(2,7)= 0.21035877343246316334D-02
         E(3,1)=-0.39890571772473709759D+00
         E(3,2)= 0.26819725655216894347D+00
         E(3,3)=-0.82551711648854471247D-01
         E(3,4)=-0.85516559106259630212D+00
         E(3,5)= 0.24433810515772642570D+00
         E(3,6)= 0.10234155624049009806D+01
         E(3,7)= 0.25115745967236579242D-01
         E(4,1)=-0.40964796048052939224D+00
         E(4,2)= 0.29949323098224574487D+00
         E(4,3)=-0.13867460566101912494D+00
         E(4,4)=-0.98859300714628940382D+00
         E(4,5)= 0.24671351779481625627D+00
         E(4,6)= 0.12912760231350872304D+01
         E(4,7)= 0.13241134766742798418D+00
         SM(1)= 0.00000000000000000000D+00
         SM(2)= 0.10000000000000000000D+01
         SM(3)= 0.16500000000000000000D+01
         AM(1)= 0.10806374869244001787D+04
         AM(2)=-0.66008818661284690206D+03
         AM(3)= 0.61810154357557529566D+03
         AM(4)=-0.31341427826212857229D+03
         AM(5)=-0.10187174765625000000D+02
         AM(6)= 0.31173050390625000000D+02
         AM(7)= 0.00000000000000000000D+00
      END IF
      IF (NS.EQ.6) THEN
         C(1)= 0.33765242898423986094D-01
         C(2)= 0.16939530676686774317D+00
         C(3)= 0.38069040695840154568D+00
         C(4)= 0.61930959304159845432D+00
         C(5)= 0.83060469323313225683D+00
         C(6)= 0.96623475710157601391D+00
         B(1)= 0.85662246189585172520D-01
         B(2)= 0.18038078652406930378D+00
         B(3)= 0.23395696728634552369D+00
         B(4)= 0.23395696728634552369D+00
         B(5)= 0.18038078652406930378D+00
         B(6)= 0.85662246189585172520D-01
         BC(1)= 0.82769839639769234611D-01
         BC(2)= 0.14982512785597570103D+00
         BC(3)= 0.14489179419935320895D+00
         BC(4)= 0.89065173086992314743D-01
         BC(5)= 0.30555658668093602753D-01
         BC(6)= 0.28924065498159379092D-02
         AA(1,1)= 0.90625420195651151857D-03
         AA(1,2)=-0.72859711612531400024D-03
         AA(1,3)= 0.79102695861167691135D-03
         AA(1,4)=-0.70675390218535384182D-03
         AA(1,5)= 0.45647714224056921122D-03
         AA(1,6)=-0.14836147050330408643D-03
         AA(2,1)= 0.11272367531794365387D-01
         AA(2,2)= 0.39083482447840698486D-02
         AA(2,3)=-0.14724868010943911900D-02
         AA(2,4)= 0.10992669056588431310D-02
         AA(2,5)=-0.67689040729401428165D-03
         AA(2,6)= 0.21677950347174141516D-03
         AA(3,1)= 0.30008019623627547434D-01
         AA(3,2)= 0.36978289259468146662D-01
         AA(3,3)= 0.65490339168957822692D-02
         AA(3,4)=-0.16615098173008262274D-02
         AA(3,5)= 0.84753461862041607649D-03
         AA(3,6)=-0.25877462623437421721D-03
         AA(4,1)= 0.49900269650650898941D-01
         AA(4,2)= 0.82003427445271620462D-01
         AA(4,3)= 0.54165111295060067982D-01
         AA(4,4)= 0.65490339168957822692D-02
         AA(4,5)=-0.11352871017627472322D-02
         AA(4,6)= 0.28963081055952389031D-03
         AA(5,1)= 0.68475836671617248304D-01
         AA(5,2)= 0.11859257878058808400D+00
         AA(5,3)= 0.10635984886129551097D+00
         AA(5,4)= 0.47961474042181382443D-01
         AA(5,5)= 0.39083482447840698486D-02
         AA(5,6)=-0.34600839001342442657D-03
         AA(6,1)= 0.79729071619449992615D-01
         AA(6,2)= 0.14419100392702230613D+00
         AA(6,3)= 0.13628542646896576408D+00
         AA(6,4)= 0.81956586217401900627D-01
         AA(6,5)= 0.23736460480774324642D-01
         AA(6,6)= 0.90625420195651151857D-03
         E(1,1)=-0.16761132335280609813D-01
         E(1,2)= 0.10201050166615899799D-01
         E(1,3)=-0.58593121685075943100D-02
         E(1,4)=-0.11907383391366998251D-03
         E(1,5)= 0.10615611118132982241D-01
         E(1,6)=-0.30692054230989138447D-01
         E(1,7)= 0.10615182045216224925D-01
         E(1,8)= 0.22586707045496892369D-01
         E(1,9)=-0.16931992776201068110D-04
         E(2,1)= 0.10671755276327262128D-01
         E(2,2)=-0.51098203653251450913D-02
         E(2,3)= 0.16062647299186369205D-03
         E(2,4)= 0.64818802653621866868D-02
         E(2,5)=-0.12132386914873895089D-01
         E(2,6)=-0.99709737725909584834D-02
         E(2,7)=-0.70287093442894942752D-02
         E(2,8)= 0.31243249755879001843D-01
         E(2,9)= 0.31763603839792897936D-04
         E(3,1)=-0.40875203230945019464D+00
         E(3,2)= 0.28214948905763253599D+00
         E(3,3)=-0.22612660499718519054D+00
         E(3,4)= 0.13640993962985420478D+00
         E(3,5)= 0.15888529591697266925D+00
         E(3,6)=-0.11667863471317749710D+01
         E(3,7)= 0.25224964119340060668D+00
         E(3,8)= 0.10440940643938620983D+01
         E(3,9)= 0.33914722176493324285D-03
         E(4,1)=-0.29437531285359759661D+01
         E(4,2)= 0.20017220470127690267D+01
         E(4,3)=-0.15383035791443948798D+01
         E(4,4)= 0.78114323215109899716D+00
         E(4,5)= 0.13930345104184182146D+01
         E(4,6)=-0.75958281612589849630D+01
         E(4,7)= 0.18220129530415584951D+01
         E(4,8)= 0.62663163493155487560D+01
         E(4,9)= 0.54279630166374655267D-02
         E(5,1)=-0.79572842006457093076D+01
         E(5,2)= 0.53527892762707449170D+01
         E(5,3)=-0.40049139768467199697D+01
         E(5,4)= 0.18326058141135591515D+01
         E(5,5)= 0.39753886181058367500D+01
         E(5,6)=-0.19423696478604790213D+02
         E(5,7)= 0.49362128400107292627D+01
         E(5,8)= 0.15601708062381928560D+02
         E(5,9)= 0.32142123424873719685D-01
         E(6,1)=-0.78463118056075171475D+01
         E(6,2)= 0.53580869574441241664D+01
         E(6,3)=-0.41476905275607763365D+01
         E(6,4)= 0.21275912797813913113D+01
         E(6,5)= 0.37642416878253538582D+01
         E(6,6)=-0.20329681631523484613D+02
         E(6,7)= 0.48515418060343387549D+01
         E(6,8)= 0.16604467346259915039D+02
         E(6,9)= 0.84559690262225766975D-01
         SM(1)= 0.00000000000000000000D+00
         SM(2)= 0.10000000000000000000D+01
         SM(3)= 0.17500000000000000000D+01
         AM(1)= 0.58080578375796358720D+05
         AM(2)=-0.33214989339522861968D+05
         AM(3)= 0.28376088288312020853D+05
         AM(4)=-0.27923430684614999462D+05
         AM(5)= 0.29743005589491042677D+05
         AM(6)=-0.15525927919158826444D+05
         AM(7)=-0.27700591278076171875D+03
         AM(8)= 0.73086943817138671875D+03
         AM(9)= 0.00000000000000000000D+00
      END IF
C
            HSTEP2=HSTEP*HSTEP
            DO IS=1,NS
              B(IS)=HSTEP*B(IS)
              BC(IS)=HSTEP2*BC(IS)
              C(IS)=HSTEP*C(IS)
              DO JS=1,NS
                AA(IS,JS)=HSTEP2*AA(IS,JS)
                E(IS,JS)=HSTEP2*E(IS,JS)
              END DO
            END DO
            DO IM=1,NM
              DO IS=1,NS
                E(IS,NS+IM)=HSTEP2*E(IS,NS+IM)
              END DO
              AM(NS+IM)=HSTEP*AM(NS+IM)
            END DO
C
      RETURN
      END
