% Driver for a non-linear RLC circuit 
% The serial circuit: R, L constant and a diode (rectfier type 1N4001)
% Model for the diode: A non constant capacitor and a non constant
% resistance in paralell.
% Diode current:
% a) Id = Is*(exp(U/(n*UT)-1)  with arameters 
%    Is =  4.8e-9   and  Ut =  25.84e-3 [V].
%    n = 1.4 cf Chatelin PPUR (not important).
% b) The capacity is a numerical model of the mesurement given in 
%    IEEE TRANSACTIONS ON ELECTRON DEVICES, VOL. 53, NO. 3, MARCH 2006:
%    C(U) = C0 + C1*exp(alpha *(U-Vbi))
%    For the 1N4001 diode, C0 = 10 pF, C1 = 120 pF, alpha = 0.5 
%    et Vbi = 0.75 (Vbi not very important)
% Differential equations:
% dU/dt = ( I -Is*(exp(U/(n*UT)-1) ) / ( C0 + C1*exp(U/(nUt) )
% dI/dt = (Ug0*sin(w*t) - R*I - U)/L
% U is the diode potential
% I is the total cirduit current
% The diode is "negative" polarized 
% This model schows under harmonic bifurcations
% ---------------------------
% See 
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%    
% See also  http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
%  Copyright (C) 2015/02, Denis Bichsel <dbichsel@infomaniak.ch>
% ---------------------------

clc
close all
clear all

addpath D:\RDPSolver

ProblemName = 'RLC_Diode';

% Function 
FcnDef = @RLC_Diode;

% Diode parameters
Diode.Ut    = 25.84e-3;     % [V]   (depend on temprature Ut = (kb*T/q))
Diode.n     = 1.4;          % cf. Dispositif  semi-conducteurs  Chatelain 
                            % PPUR 1986 page 62
% Current                  
Diode.Is    = 4.8e-9;       % Saturation Current [A]
% Capacit
Diode.C0    = 10e-12;       % [F]
Diode.C1    = 160e-12;      % [F]
Diode.Vbi   = 0.75;         % [V] Built in Voltage
Diode.alpha = 0.5;          % [1/V]

% Circuit parameters
Circuit.R = 22;             % [Ohm]
Circuit.L = 0.01;           % [H]

% Generator parameters
Gen.U0  = 2;                % Peak differential potential
% And now, the frequency which is chosen very near the resonant frequency
% of the L-C circuit with no excitation.
U = 0;
[Ir,dIrdU,C,dCdU,d2CdU2] = RLC_Diode_IC(U,Diode);
Gen.w0 = 1/sqrt(Circuit.L*C);                                       
Gen.T  = 2*pi/Gen.w0;

% Initial conditions
y0                = [0.0;0.0];  

% Diode caracteristics.
Umin   = -0.9;
Umax   = 0.5;
dU     = 0.001;
U      = Umin:dU:Umax;
[Ir,C] = RLC_Diode_IC(U,Diode);

Fig = 1;
figure(Fig); Fig = Fig+1;
subplot(2,1,1);
plot(U,Ir);
grid on
title('Potential - Current ( diode 1N4001 )');
xlabel('Potentiel   [V]');
ylabel('Current  [ A ]');
subplot(2,1,2);
semilogy(U,C);
grid on
title('Potential - Capacity ( diode 1N4001 )');
xlabel('Potentiel   [V]');
ylabel('Capacity  [ F ]');
%------------------

% Parameters
OdeCoeff{1} = Diode;
OdeCoeff{2} = Gen;
OdeCoeff{3} = Circuit;

Ti      = 0;                       
Tf      = 50*Gen.T; 
dt        = 0.01*Gen.T;
tspan     = [Ti :dt: Tf]';
%------------------

tol        = 1e-6;
RelTolDef  = tol;
AbsTolDef  = tol;

InfoStatDef   = true;
InfoDynDef    = true;

NbrStgDef    = 3; 
MinNbrStgDef = 3;
MaxNbrStgDef = 7;

options = [];
options = rdpset(options,'RelTol',RelTolDef);
options = rdpset(options,'AbsTol',AbsTolDef);
options = rdpset(options,'NbrStg',NbrStgDef);
options = rdpset(options,'MinNbrStg',MinNbrStgDef);
options = rdpset(options,'MaxNbrStg',MaxNbrStgDef);

tic
[tout,yout,Stats] = radau(FcnDef,tspan,y0,options,OdeCoeff);     
T_radau = toc
 
InfoStat = Stats.Stat;
InfoDyn  = Stats.Dyn;

figure(Fig); Fig = Fig+1;
subplot(3,1,1)
plot(tout,yout(:,1))
grid on
title(['RLC Diode, U0 = ',num2str(Gen.U0),'  Bifurcation']);
ylabel('Diode Voltage Ud')
xlabel('Time')
subplot(3,1,2)
plot(tout,yout(:,2),'r')
ylabel('Derivative of the Diode Voltage d(Ud)/(dt)'' ')
xlabel('Time')
grid on
title(['RLC_Diode, U0 = ',num2str(Gen.U0)]);
subplot(3,1,3)
plot(tout,Gen.U0*sin(Gen.w0*tout))
grid on
title(['Generator Voltage  U0 = ',num2str(Gen.U0)]);
ylabel('Generator Voltage Ud')
xlabel('Time')

figure(Fig); Fig = Fig+1;
plot(yout(:,1),yout(:,2))
grid on
title(['RLC-Diode, U0 = ',num2str(Gen.U0)]);
ylabel('Ud'' ')
xlabel('Ud')

if InfoDynDef
  figure   
  subplot(4,1,1)
  plot(tout,real(yout(:,1)),'b')
  grid on
  hold on
  plot(tout,real(yout(:,1)),'.r')
  title('Oscillator component no 1')
    
  subplot(4,1,2)
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept);
  hold on
  grid on
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept,'.k');
  plot(InfoDyn.hreject_t,InfoDyn.hreject,'xr')
  xlim([tspan(1) tspan(end)])
  title('Length of steps in function of time')
    
  subplot(4,1,3)
  plot(InfoDyn.Newt_t,InfoDyn.NewtNbr)
  grid on
  xlim([tspan(1) tspan(end)])
  title('Newton iteration number in function of time')
    
  subplot(4,1,4)
  plot(InfoDyn.Newt_Step,InfoDyn.NewtNbr)
  hold on
  plot(InfoDyn.Jac_Step,0.5,'.b');
  title('Newton iteration number in function of steps and Jacobian')
    
  figure
  plot(InfoDyn.NbrStg_t,InfoDyn.NbrStg,'or')
  grid on
  figure
  plot(InfoDyn.NbrStg_Step,InfoDyn.NbrStg,'ob')
  grid on
  Nbr1 = length(find(InfoDyn.NbrStg == 1));
  Nbr3 = length(find(InfoDyn.NbrStg == 3));
  Nbr5 = length(find(InfoDyn.NbrStg == 5));
  Nbr7 = length(find(InfoDyn.NbrStg == 7));
  NbrStg_1_3_5_7 = [Nbr1,Nbr3,Nbr5,Nbr7]
end

% Statistique
if InfoStatDef
  InfoStat = InfoStat
end

options = [];
options = odeset(options,'RelTol',RelTolDef);
options = odeset(options,'AbsTol',AbsTolDef);

tic
[tout,yout,Stat] = ode15s(FcnDef,tspan,y0,options,OdeCoeff);     
T_ode15s = toc

figure(Fig); Fig = Fig+1;
subplot(3,1,1)
plot(tout,yout(:,1))
grid on
title(['RLC Diode, U0 = ',num2str(Gen.U0),'  Bifurcation']);
ylabel('Diode Voltage Ud')
xlabel('Time')
subplot(3,1,2)
plot(tout,yout(:,2),'r')
ylabel('Derivative of the Diode Voltage d(Ud)/(dt)'' ')
xlabel('Time')
grid on
title(['RLC_Diode, U0 = ',num2str(Gen.U0)]);
subplot(3,1,3)
plot(tout,Gen.U0*sin(Gen.w0*tout))
grid on
title(['Generator Voltage  U0 = ',num2str(Gen.U0)]);
ylabel('Generator Voltage Ud')
xlabel('Time')


