% Driver for a non-linear RLC circuit 
% Model
% A self induction L is serial with a condensator C which is in parallel
% with a tunnel diode. 
%                |---- C  ----|
%    |---- L ----|            | ----|
%    |           |---- TD ----|     |
%    |                              |
%    |---------------- Ug ----------|
% A sinusoidal generator Ug0*sin(w*t) excite the circuit.
% Model of the tunnel diode (Voltage - Current)
%  Id = Ip *(Ud/Up) .* exp(1.0 - Ud/Up) + ...
%       Iv*exp(-A2*Uv)*(exp(A2*Ud)-1)  + ...
%       Is * (exp(Ud/Ut) - 1.0);
% Ut = 25.84e-3;  Up = 0.065; Ip = 4.7e-3; Uv = 0.355; Iv = 0.45e-3;   
% Is = 50e-12; A2 = 15;
% Moodel of the circuit
% First Kirchhoff law
%   L*dI/dt + Ud = Ug
% Second Kirchhoff law
%   I = Id + Ic     
%   Ic = C*Ud'   then
%   L*(CUd'' '+ Id') + Ud = Ug
%   and in usual form:
%   Ud'' = 1/(LC)*(Ug-Ud-L*dId/dUd * Ud'
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%     
% See also http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
% ---------------------------

clc
close all
clear all

addpath D:\RDPSolver

ProblemName = 'RLCDiodeTunnel';

% Function and jacobian
FcnDef = @RLCDiodeTunnel;
JacDef = @RLCDiodeTunnelJac;

% Parameters
% Tunneldiode parameters
Ut = 25.84e-3;     
Up = 0.065; 
Ip = 4.7e-3; 
Uv = 0.355; 
Iv = 0.45e-3;   
Is = 50e-12;
A2 = 15;

% Circuit parameters
L    = 0.1;
C    = 1e-6;

% Generator parameters
Ug0  = 6.853;

% Ug0  = 2;      % Priode 1
% Ug0  = 5.0;    % Priode 2
% Ug0  = 6.0;    % Priode 2
% Ug0  = 6.5;    % Priode 2
% Ug0  = 6.7;    % Priode 4
% Ug0  = 6.8;    % Priode 4
% Ug0  = 6.85;   % Priode 8
% Ug0  = 6.851;  % Priode 8
% Ug0  = 6.852;  % Priode 8
% Ug0  = 6.853;  % Priode 16
% UG0  = 6.855;  % Priode chaos
% Ug0  = 6.86;   % Priode chaos
% Ug0  = 6.9;    % Priode chaos
% Ug0  = 7.0;    % Priode 5
% Ug0  = 7.5;    % Priode 2
% Ug0  = 8;      % Priode 1
% Ug0  = 10.2;   % Priode 1

% Parameters
OdeCoeff{1}  = Ut;
OdeCoeff{2}  = Up ;
OdeCoeff{3}  = Ip;
OdeCoeff{4}  = Uv;
OdeCoeff{5}  = Iv;
OdeCoeff{6}  = Is;
OdeCoeff{7}  = A2;
OdeCoeff{8}  = L;
OdeCoeff{9}  = C;
OdeCoeff{10} = Ug0;

% ---------------------------
% Tunneldiode caracteristics.
Umin = -0.01;
Umax = 0.45;
dU   = 0.001;
Ud   = [Umin:dU:Umax]';

Id = RLCDiodeTunnel_Ir_dIr_d2Ir(Ud,OdeCoeff);

Fig = 1;
figure(Fig); Fig = Fig+1;
plot(Ud,Id);
hold on
grid on
plot(Up,Ip,'or')
plot(Uv,Iv,'or')
title('Tunnel diode voltage - current caracteristic');
xlabel('Voltage  [V]');
ylabel('Currentt [ A ]');
%----------------------------

Ti     = 0;
F      = 1/(2*pi*sqrt(L*C));
NbrPer = 150;
T      = 1/F;
Tf     = NbrPer * T;
tspan  = [Ti Tf];

% Initial conditions
y0  = [0;0];

tol        = 1e-10;
RelTolDef  = tol;
AbsTolDef  = tol;

InfoStatDef   = true;
InfoDynDef    = true;

NbrStgDef    = 3; 
MinNbrStgDef = 3;
MaxNbrStgDef = 7;


options = [];
options = rdpset(options,'RelTol',RelTolDef);
options = rdpset(options,'AbsTol',AbsTolDef);
options = rdpset(options,'NbrStg',NbrStgDef);
options = rdpset(options,'MinNbrStg',MinNbrStgDef);
options = rdpset(options,'MaxNbrStg',MaxNbrStgDef);
options = rdpset(options,'JacFcn',JacDef);

warning on
tic
[tout,yout,Stats] = radau(FcnDef,tspan,y0,options,OdeCoeff);    
toc

InfoStat = Stats.Stat;
InfoDyn  = Stats.Dyn;

Ud = yout(:,1);

figure(Fig); Fig = Fig + 1;
subplot(2,1,1);
plot(tout/T,Ud,'-')
grid on
title(['Tunnel diode RLC  Voltage,   Ugen0 = ',num2str(Ug0)]);
xlabel('Time [s]');
ylabel('Voltage [V]');

[Id,dId] = RLCDiodeTunnel_Ir_dIr_d2Ir(Ud,OdeCoeff);

subplot(2,1,2);
plot(tout/T,Id,'-')
grid on
title(['Tunnel diode RLC  Current,   Ugen0 = ',num2str(Ug0)]);
xlabel('Time [s]');
ylabel('Current [ A ]');

figure(Fig); Fig = Fig+1;
[M,N] = size(yout);
plot(yout(ceil(M/2):M,1), Id(ceil(M/2):M,1))
grid on
title(['Current versus Voltage for time larger than : ',num2str(tout(ceil(M/2)))] );

if InfoDynDef
  figure   
  subplot(4,1,1)
  plot(tout,real(yout(:,1)))
  grid on
  title('Tunnel Diode Voltage ')
    
  subplot(4,1,2)
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept);
  hold on
  grid on
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept,'.k');
  plot(InfoDyn.hreject_t,InfoDyn.hreject,'xr')
  xlim([tspan(1) tspan(end)])
  title('Length of steps in function of time')
    
  subplot(4,1,3)
  plot(InfoDyn.Newt_t,InfoDyn.NewtNbr)
  grid on
  xlim([tspan(1) tspan(end)])
  title('Newton iteration number in function of time')
    
  subplot(4,1,4)
  plot(InfoDyn.Newt_Step,InfoDyn.NewtNbr)
  hold on
  plot(InfoDyn.Jac_Step,0.5,'.b');
  title('Newton iteration number in function of steps and Jacobian')
    
  figure
  subplot(2,1,1)
  plot(InfoDyn.NbrStg_t,InfoDyn.NbrStg,'or')
  grid on
  title('Number of Stages versus Time')
  subplot(2,1,2)
  plot(InfoDyn.NbrStg_Step,InfoDyn.NbrStg,'ob')
  grid on
  title('Number of Stages versus Steps Number')
  Nbr1 = length(find(InfoDyn.NbrStg == 1));
  Nbr3 = length(find(InfoDyn.NbrStg == 3));
  Nbr5 = length(find(InfoDyn.NbrStg == 5));
  Nbr7 = length(find(InfoDyn.NbrStg == 7));
  NbrStg_1_3_5_7 = [Nbr1,Nbr3,Nbr5,Nbr7]
end
% Statistique
if InfoStatDef
  InfoStat = InfoStat
end



