% ---------------------------
% Circuit RLC non-linaire, la bobine a un noyau de ferite.
%
% Ce programme rsoud les quations diffrentielles dcrivant un cicuit
% oscillant RLC srie dont le coefficient d'auto-induction dpend du 
% courant. Il s'agit par exemple d'une bobine construite sur un noyau
% de ferrite. Le coefficient d'auto-induction dcroit fortement en 
% fonction du courant.
% Le modle :
%   B ~ L0*mu*I0*tanh(I/I0) + L0*I
% La drive de tanh: (tanh)' = 1/cosh^2 ainsi:
%   L(I) = L0(1+mu/cosh(I/I0)^2)
% L0 est le coefficient de la bobine construite sans ferrite
% mu est la permabilit de la ferrite  faible courant (champs).
% Les quations diffrentielles sont:
% d(LI)/dt + RI + U = Ug% Le modle :
% U est la diffrence de potentiel aux bornes du condensateur
% I est le courant total du circuit
% ---------------------------
% Voir 
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%    
%    E. Hairer C. Lubich G.Wanner
%    Geometric Numerical Integration.
%    Structure-Preserving Algorithms for Ordinary Differential Equations 	 
%	   Springer Verlag ISBN  3-540-43003-2  ISSN 0-179-3632
%    ISBN 3-540-43003-2,  ISSN 0179-3632
%     
%    Voir galement http://www.unige.ch/~hairer/software.html
% ---------------------------
%  Copyright (C) 2015/02, Denis Bichsel <dbichsel@infomaniak.ch>
% 
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
% ---------------------------

clear all
close all
clc
warning on

addpath D:\RDPSolver

NomProbleme = 'RLC-Ferrite';

global L0 mu I0 R C U0 w0

% Paramtre de la bobine 
L0 = 1e-3;      % [H] Henri
mu = 100;          
I0 = 0.02;     % [A]
L  = L0*(1+mu);
% Paramtres du Circuit 
R  = 3;        % [Ohm]
C  = 7.5e-6;   % [F]

% Gnrateur
U0 = 15.1;          % Tension de crte du gnrateur
w0 = 1/sqrt(L*C);   % gale  la rsonance du circuit faiblement excit
T  = 2*pi/w0 ;      % Priode du gnrateur

% U0 = 10      Priode 1     % R = 3.0,   C = 7.5e-6,  tol = 1e-10
% U0 = 15      Priode 2     % L0 = 1e-3, mu = 100,    I0 = 0.02
% U0 = 15.1    Priode 2     % Avec la fonction jacobienne
% U0 = 15.2    Priode 4
% U0 = 15.202  Priode 4
% U0 = 15.22   Priode 4
% U0 = 15.25   Priode 4
% U0 = 15.27   Priode 8
% U0 = 15.28   Priode 8
% U0 = 15.285  Priode 32
% U0 = 15.3    Priode chao
% U0 = 15.5    Priode cha0
% U0 = 16      Priode chao
% U0 = 20      Priode chao

% U0 = 13.5    Priode 2     % R = 3.0 C = 7.5e-6     
                             % L0 = 1e-3, mu = 100, I0 = 0.018

% Conditions initiales
y0 = [0.0;0.0];    % Conditions initiales

% Visualisation de la caractristiques auto-inductive de la bobine
Imin    = -5*I0;
Imax    =  5*I0;
dI      =  0.001;
I       = Imin:dI:Imax;
[Phi,L] = RLC_Ferrite_Phi_L(I);

Fig = 1;
figure(Fig); Fig = Fig+1;
subplot(2,1,1)
plot(I,Phi);
grid on
title('Caractristique Courant - Flux ');
xlabel('Courant   [A]');
ylabel(' Phi  [HA]');

subplot(2,1,2)
plot(I,L);
grid on
title('Caractristique Courant - Auto-induction ');
xlabel('Courant  [A]');
ylabel(' L  [H]');

% Evolution temporelle du circuit excit
tmin      = 0;                       
tmax      = 100*T; 
dt        = 0.01*T;
tspan     = [tmin:dt:tmax]';

% Equation diffrentielle
OdeFcn      = @RLC_Ferrite;

% Jacobien
JacFct      = true;            % Vrai --> La fonction est donne,
                               % false --> le jacobien est calcul                                % numriquement
                                
% Paramtres du Solver 
tol        = 1e-10;
RelTolDef  = tol;
AbsTolDef  = tol;

% InfoStatDef   = true;
% InfoDynDef    = true;

NbrStgDef    = 5; 
MinNbrStgDef = 3;
MaxNbrStgDef = 7;
options = [];
options = rdpset(options,'RelTol',RelTolDef);
options = rdpset(options,'AbsTol',AbsTolDef);
options = rdpset(options,'NbrStg',NbrStgDef);
options = rdpset(options,'MinNbrStg',MinNbrStgDef);
options = rdpset(options,'MaxNbrStg',MaxNbrStgDef);
if JacFct
  JacDef  = @RLC_FerriteJac;  % Fonction dfinissant le jacobien
  options = rdpset(options,'JacFcn',JacDef);
end


tic
[t,y] = radau(OdeFcn,tspan,y0,options);
T_radau = toc

NbrPer = t/T;
figure(Fig); Fig = Fig+1;
subplot(3,1,1)
plot(NbrPer,y(:,1))
grid on
title(['RLC-Ferrite, U0 = ',num2str(U0)]);
ylabel('Tension condensateur U')
xlabel('Temps')
subplot(3,1,2)
plot(NbrPer,C*y(:,2))
ylabel('Courant I ')
xlabel('Temps')
grid on
title(['RLC-Ferrite, U0 = ',num2str(U0)]);
subplot(3,1,3)
plot(NbrPer,U0*sin(w0*t))
grid on
title(['Tension du gnrateur  U0 = ',num2str(U0)]);






