function Jac = BrusselatorDiffusifJac(t,uv,OdeCoeff)
% Differential equations:
% du/dt = g1 = A + u^2* v -(B+1)*u + alpha*d2u/dx2
% dv/dt = g2 = B*u - U^2*v + a*d2v/dx2
% Parameters  A, B and alpha
% To get a 5 bands matrix, the equations are interleaved
% Set y0    = [u0(1);v0(1);u0(2);v0(2); ... ].
%     dy/dt = [du(1)/dt; dv(1)/dt;      ... ].
% The jacobian is a sparse matrix (5 bands).
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%    
% See also http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
% ---------------------------
xspan    = OdeCoeff{1};
A        = OdeCoeff{2};
B        = OdeCoeff{3};
alpha    = OdeCoeff{4};

dx    = xspan(2) - xspan(1);
adx2  = alpha/dx^2;

N     = length(uv);

Jac   = sparse(zeros(N));

% First two lines
Jac(1,1) = 2*uv(1)*uv(2) - (B+1) - 2*adx2;
Jac(1,2) = uv(1)^2;
Jac(1,3) = adx2;

Jac(2,1) = B - 2*uv(1)*uv(2);
Jac(2,2) = - uv(1)^2 - 2*adx2; 
Jac(2,4) = adx2;

for i = 3:2:N-3
  Jac(i,i-2) = adx2;
  Jac(i,i)   = 2*uv(i)*uv(i+1) - (B+1) - 2*adx2;
  Jac(i,i+1) = uv(i)^2;
  Jac(i,i+2) = adx2;
end
for i = 4:2:N-1
  Jac(i,i-2) = adx2; 
  Jac(i,i-1) = B - 2*uv(i-1)*uv(i);
  Jac(i,i)   = - uv(i-1)^2 - 2*adx2; 
  Jac(i,i+2) = adx2;
end

% Last two lines
Jac(N-1,N-3) = adx2;
Jac(N-1,N-1) = 2*uv(N-1)*uv(N) - (B+1) - 2*adx2;
Jac(N-1,N)   = uv(N-1)^2;

Jac(N,N-2)   = adx2;
Jac(N,N-1)   = B - 2*uv(N-1)*uv(N);
Jac(N,N)     = - uv(N-1)^2 - 2*adx2; 
    
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  