% Driver for the non-diffusive brusselator 
% Differential equations:
% dy1/dt = A + y1.^2.*y2 - (B+1)*y1;
% dy2/dt = B*y1 - y1.^2.*y2;
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%    
% See also http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
% ---------------------------

clc
close all
clear all

addpath D:\RDPSolver

ProblemName = 'Non-diffusive brusselator';

% Function and jacobian
FcnDef = @BrusselatorFcn;
JacDef = @BrusselatorJac;

% Parameters
Bru_A = 2; 
Bru_B = 8.533;

% Initial conditions 
y0    = [1.0; 4.2665]; 

Ti    = 0;
Tf    = 20;
tspan = [Ti  Tf];

OdeCoeff{1} = Bru_A;
OdeCoeff{2} = Bru_B;

tol        = 1e-6;
RelTolDef  = tol;
AbsTolDef  = tol;

InfoStatDef    = true;
InfoDynDef     = true;

NbrStgDef    = 3; 
MinNbrStgDef = 3;
MaxNbrStgDef = 7;


options = [];
options = rdpset(options,'RelTol',RelTolDef);
options = rdpset(options,'AbsTol',AbsTolDef);
options = rdpset(options,'NbrStg',NbrStgDef);
options = rdpset(options,'MinNbrStg',MinNbrStgDef);
options = rdpset(options,'MaxNbrStg',MaxNbrStgDef);
options = rdpset(options,'JacFcn',JacDef);

% First, solution via odeplot
tic
radau(FcnDef,tspan,y0,options,OdeCoeff);    
T_radau = toc

% Second, soltion via normal output
tic
[tout,yout,Stats] = radau(FcnDef,tspan,y0,options,OdeCoeff);    
T_radau = toc

InfoStat = Stats.Stat;
InfoDyn  = Stats.Dyn;

figure
plot(tout,real(yout(:,1)),'b')
hold on
plot(tout,real(yout(:,1)),'.b')
grid on
plot(tout,real(yout(:,2)),'r')
plot(tout,real(yout(:,2)),'.r')
title('Brusselator, u  in blue   v in red')
   
if InfoDynDef
  figure    
  subplot(4,1,1)
  plot(tout,real(yout(:,1)),'b')   
  hold on
  plot(tout,real(yout(:,1)),'.b')
  grid on
  plot(tout,real(yout(:,2)),'r')
  plot(tout,real(yout(:,2)),'.r')
  title('Brusselator, u  in blue   v in red')
    
  subplot(4,1,2)
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept);
  hold on
  grid on
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept,'.k');
  plot(InfoDyn.hreject_t,InfoDyn.hreject,'xr')
  xlim([0 2])
  title('Length of steps in function of time')
    
  subplot(4,1,3)
  plot(InfoDyn.Newt_t,InfoDyn.NewtNbr)
  grid on
  xlim([0 2])
  title('Newton iteration number in function of time')
    
  subplot(4,1,4)
  plot(InfoDyn.Newt_Step,InfoDyn.NewtNbr)
  hold on
  plot(InfoDyn.Jac_Step,0.5,'.b');
  title('Newton iteration number in function of steps and jacobian')
    
  figure
  subplot(2,1,1)
  plot(InfoDyn.NbrStg_t,InfoDyn.NbrStg,'or')
  grid on
  title('Number of Stages versus Time')
  subplot(2,1,2)
  plot(InfoDyn.NbrStg_Step,InfoDyn.NbrStg,'ob')
  grid on
  title('Number of Stages versus Steps Number')
  Nbr1 = length(find(InfoDyn.NbrStg == 1));
  Nbr3 = length(find(InfoDyn.NbrStg == 3));
  Nbr5 = length(find(InfoDyn.NbrStg == 5));
  Nbr7 = length(find(InfoDyn.NbrStg == 7));
  NbrStg_1_3_5_7 = [Nbr1, Nbr3, Nbr5, Nbr7]
end
if InfoStatDef  
  InfoStat = InfoStat
end

options = [];
options = rdpset(options,'RelTol',RelTolDef);
options = rdpset(options,'AbsTol',AbsTolDef);
options = rdpset(options,'JacFcn',JacDef);

tic
[tout,yout,Stats] = ode15s(FcnDef,tspan,y0,options,OdeCoeff);    
T_ode15s = toc


figure
plot(tout,real(yout(:,1)),'b')
hold on
plot(tout,real(yout(:,1)),'.b')
grid on
plot(tout,real(yout(:,2)),'r')
plot(tout,real(yout(:,2)),'.r')
title('Brusselator, u  in blue   v in red')
