% Driver for Ex1 of Mathworks
%  This is a singular perturbation problem involving Michaelis-Menten
%  kinetics.  It is studied in Chapter 10 of C. Lin and L. Segal, 
%  Mathematics Applied to Deterministic Problems in the Natural Sciences,
%  SIAM, Philadelphia, 1988.  The problem is
%
%          s0' = -s0 + (s0 + kappa - lambda) * c0    s0(0) = 1   
%  epsilon*c0' =  s0 - (s0 + kappa) * c0             c0(0) = 1
%
%  The parameters used here are taken from p. 317:
%         kappa = 1, lambda = 0.375  and    0 <= t <= 1.2.
%
%  First we write the equations in the form M(t)y' = f(t,y):
%  Let y(1) = s0(t) and y(2) = c0(t).  M(t) = diag{1, epsilon}.  
%
%  This is an ODE for epsilon > 0.  As epsilon -> 0, there is a boundary
%  layer at the origin and for t > 0, the solution tends to an "outer 
%  solution" which satisfies a DAE with epsilon = 0.
%
% Copyright 2004, The MathWorks, Inc.
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%    
% See also http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
%  Copyright (C) 2015/02, Denis Bichsel <dbichsel@infomaniak.ch>
% ---------------------------

clc
close all
clear all

addpath D:\RDPSolver

ProblemName = 'Ex1';

% Function and jacobian
FcnDef  = @Ex1Fcn;
JacDef  = @Ex1Jac;
MassDef = @Ex1Mass;

% Problem parameters 
kappa   = 1;
lambda  = 0.375; 
epsilon = 0.01;

% Initial conditions 
y0 = [1; 1];

Ti    = 0;
Tf    = 1.2;
tspan = [Ti Tf];

OdeCoeff(1) = kappa;
OdeCoeff(2) = lambda;
OdeCoeff(3) = epsilon;

tol        = 1e-6;
RelTolDef  = tol;
AbsTolDef  = tol;

NbrStgDef    = 5; 
MinNbrStgDef = 5;
MaxNbrStgDef = 7;

options = [];
options = rdpset(options,'NbrStg',NbrStgDef);
options = rdpset(options,'MinNbrStg',MinNbrStgDef);
options = rdpset(options,'MaxNbrStg',MaxNbrStgDef);
options = rdpset(options,'RelTol',RelTolDef);
options = rdpset(options,'AbsTol',AbsTolDef);
options = rdpset(options,'JacFcn',JacDef);
options = rdpset(options,'MassFcn',MassDef);

% First solve the ODE.  Use a "large" value of epsilon so 
% that the boundary layer is visible.
tic
[t,y,Stats] = radau(FcnDef,tspan,y0,options,OdeCoeff); 
T_radau = toc

figure(1)
plot(t,y)
axis([0 1.2 0.4 1.1])
title(['Singularly perturbed ODE with epsilon = ',num2str(epsilon),'.'])
grid on

% Now solve as a DAE:
epsilon   = 0;
MassCoeff = epsilon;

y0 = [1;1/2];

tic
[tdae,ydae] = radau(FcnDef,tspan,y0,options,OdeCoeff); 
T_radau = toc

figure(2)
plot(tdae,ydae)
axis([0 1.2 0.4 1.1])
grid on
title('Outer Solution from a DAE.')

fprintf('Given initial conditions are      %6.4f %6.4f.\n',y0(1),y0(2))
fprintf('Consistent initial conditions are %6.4f %6.4f.\n',ydae(1,1),ydae(1,2))


