% Driver for the two phase plug flow problem
%  This is the two phase plug flow problem from E. Hairer, Ch. Lubich,
%  and M. Roche, The Numerical Solution of Differential-Algebraic Systems 
%  by Runge-Kutta Methods, Lecture Notes in Math. #1409, Springer, Berlin,
%  1989, specifically the normal flow example.  There are two algebraic 
%  variables.  The variables of HLR correspond to those used here as follows:
%     P <--> y(1), u <--> y(2), y <--> y(3). 
% Copyright 2004, The MathWorks, Inc.
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%    
%    E. Hairer C. Lubich M. Roche, 
%    The Numerical Solution of Differential-Algebraic Systems 
%    by Runge-Kutta Methods. 
%    Lecture Notes in Mathematics, Vol. 1409, Springer-Verlag 1989. 
%    
% See also http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
% ---------------------------

clc
clear all
close all
 
addpath D:\RDPSolver

ProblemName = 'Ex5';

% Function and jacobian
FcnDef  = @Ex5Fcn;
MassDef = @Ex5Mass;

% Parameters
R   = 45.72;
rho = 0.814;
mu  = 0.098;
b   = 0.606;
Qco = 1.1531e6;
Qa  = 2.035e5;

%  Consistent initial conditions from HLR:
y0    = zeros(3,1);
y0(1) = 1.457e8; 
y0(2) = 1.651469869370661357709407;
y0(3) = 4.877400707738074248312839;

OdeCoeff{1} = R;
OdeCoeff{2} = rho;
OdeCoeff{3} = mu;
OdeCoeff{4} = b;
OdeCoeff{5} = Qco;
OdeCoeff{6} = Qa;
OdeCoeff{7} = y0(1);

L     = 8.047e6;
tspan = [0 L];

RelTolDef = 1e-3;
AbsTolDef = 1e-6;

InfoStatDef    = true;
InfoDynDef     = true;

NbrStgDef    = 3; 
MinNbrStgDef = 3;
MaxNbrStgDef = 7;

options = [];
options = rdpset(options,'RelTol',RelTolDef);
options = rdpset(options,'AbsTol',AbsTolDef);
options = rdpset(options,'NbrStg',NbrStgDef);
options = rdpset(options,'MinNbrStg',MinNbrStgDef);
options = rdpset(options,'MaxNbrStg',MaxNbrStgDef);
options = rdpset(options,'MassFcn',MassDef);

tic
[tout,yout,Stats] = radau(FcnDef,tspan,y0,options,OdeCoeff);    
T_radau = toc

InfoStat = Stats.Stat;
InfoDyn  = Stats.Dyn;

figure(1)
plot(tout,1e-7*yout(:,1),tout,yout(:,3))
axis([9e5 L 0 15])
title('Two Phase Plug Flow--Normal Flow.')
xlabel('The first component is multiplied by 1e-7.')
grid on

if InfoDynDef
  figure
    
  subplot(4,1,1)
  plot(tout,real(yout(:,1)))   
  hold on
  plot(tout,real(yout(:,1)),'.b')
  grid on 
  plot(tout,real(yout(:,3)))
  plot(tout,real(yout(:,3)),'.r')
  title('Two Phase Plug Flow--Normal Flow.')
  xlabel('The first component is multiplied by 1e-7.')
      
  subplot(4,1,2)
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept);
  hold on
  grid on
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept,'.k');
  plot(InfoDyn.hreject_t,InfoDyn.hreject,'xr')
  xlim([0 L])
  title('Length of steps in function of time')
    
  subplot(4,1,3)
  plot(InfoDyn.Newt_t,InfoDyn.NewtNbr)
  grid on
  xlim([0 L])
  title('Newton iteration number in function of time')
    
  subplot(4,1,4)
  plot(InfoDyn.Newt_Step,InfoDyn.NewtNbr)
  hold on
  plot(InfoDyn.Jac_Step,0.5,'.b');
  title('Newton iteration number in function of steps and jacobian')
    
  figure
  subplot(2,1,1)
  plot(InfoDyn.NbrStg_t,InfoDyn.NbrStg,'or')
  grid on
  title('Number of Stages versus Time')
  subplot(2,1,2)
  plot(InfoDyn.NbrStg_Step,InfoDyn.NbrStg,'ob')
  grid on
  title('Number of Stages versus Steps Number')
  Nbr1 = length(find(InfoDyn.NbrStg == 1));
  Nbr3 = length(find(InfoDyn.NbrStg == 3));
  Nbr5 = length(find(InfoDyn.NbrStg == 5));
	Nbr7 = length(find(InfoDyn.NbrStg == 7));
  NbrStg_1_3_5_7 = [Nbr1, Nbr3, Nbr5, Nbr7]
end
if InfoStatDef  
  InfoStat = InfoStat
end
