function ex5
%  This is the two phase plug flow problem from E. Hairer, Ch. Lubich,
%  and M. Roche, The Numerical Solution of Differential-Algebraic Systems 
%  by Runge-Kutta Methods, Lecture Notes in Math. #1409, Springer, Berlin,
%  1989, specifically the normal flow example.  There are two algebraic 
%  variables.  The variables of HLR correspond to those used here as follows:
%     P <--> y(1), u <--> y(2), y <--> y(3). 

% Copyright 2004, The MathWorks, Inc.

%  Consistent initial conditions from HLR:
y0 = zeros(3,1);
y0(1)  =  1.457e8; 
y0(2)  =  1.651469869370661357709407;
y0(3)  =  4.877400707738074248312839;

%  Change the algebraic components to test initialization.
%y0(2:3) = 0.5 * y0(2:3);

L = 8.047e6;
tspan = [0 L];

M = diag([1 0 0]);
options = odeset('Mass',M,'RelTol',1e-6,'AbsTol',1e-6);

tic
[t,y] = ode15s(@ex5ode,tspan,y0,options);
t_ode15s = toc

figure(1)
plot(t,1e-7*y(:,1),t,y(:,3))
axis([9e5 L 0 15])
title('Two Phase Plug Flow--Normal Flow.')
xlabel('The first component is multiplied by 1e-7.')

%------------------------------------------------------------------------------
% Nested function -- parameter 'y0' shared with the outer function.

  function yp = ex5ode(t,y)

    yp = zeros(3,1);
    yp(1) = - y(2)^2;
    
    R = 45.72;
    rho = 0.814;
    mu = 0.098;
    b = 0.606;
    Qco = 1.1531e6;
    Qa = 2.035e5;
    
    temp1 = pi*sqrt(0.5*R/rho);
    temp2 = sqrt(0.5*R*rho);
    temp3= log(temp2 * (y(3)/mu) * y(2) - 5);
    
    yp(2) =   temp1*(R - y(3))^2 * y(2) * (2.5*temp3 + 10.5) ...
              - b*Qco - (y0(1)/y(1)) * Qco * (1 - b);
    
    yp(3) = 2*temp1*y(2)*((2.5*R*y(3) - 1.25*y(3)^2)* ...
              temp3 + 3*R*y(3) - 2.125*y(3)^2 - 13.6*R*mu/(temp2*y(2))) - Qa;
    
  end  % function ex5ode
  
%------------------------------------------------------------------------------

end  % ex5
