% Driver for the amplifier with 1 transistor 
%  This is the one transistor amplifier problem displayed on p. 377 of 
%  E. Hairer and G. Wanner, Solving Ordinary Differential Equations II
%  Stiff and Differential-Algebraic Problems, 2nd ed., Springer, Berlin,
%  1996.  It is easy enough to write this problem in semi-explicit form,
%  but it arises in the form M*u' = f(t,u) with a mass matrix that is
%  not diagonal and it is solved here in the form it arises.  The mass
%  matrix is treated as a sparse matrix to test the capability. Fig. 1.4
%  of Hairer and Wanner shows the solution on [0 0.2], but here it is
%  computed on [0 0.05] because the computation is less expensive and 
%  the nature of the solution is clearly visible on the shorter interval.
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%    
% See also http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
% ---------------------------

clc
clear all
close all
 
addpath D:\RDPSolver

ProblemName = 'Ex7';

% Function and jacobian
FcnDef  = @Ex7Fcn;
MassDef = @Ex7Mass;

% Parameters
c     = 1e-6 * [1:3];
R0    = 1000;
R     = 9000;
Beta  = 1e-6;
Ug    = 0.4;
Uf    = 0.026;
Omega = 200*pi;

% Initial conditions (they must be consistent and they are)
Ub    = 6;
u0    = zeros(5,1);
u0(2) = Ub/2;
u0(3) = u0(2);
u0(4) = Ub;

Ti = 0;
Tf = 0.2;
tspan = [Ti Tf];

OdeCoeff{1} = Ub;
OdeCoeff{2} = c;
OdeCoeff{3} = R0;
OdeCoeff{4} = R;
OdeCoeff{5} = Beta;
OdeCoeff{6} = Ug;
OdeCoeff{7} = Uf;
OdeCoeff{8} = Omega;

RelTolDef  = 1e-6;
AbsTolDef  = 1e-6;

InfoStatDef    = true;
InfoDynDef     = true;

NbrStgDef    = 3; 
MinNbrStgDef = 3;
MaxNbrStgDef = 7;

options = [];
options = rdpset(options,'RelTol',RelTolDef);
options = rdpset(options,'AbsTol',AbsTolDef);
options = rdpset(options,'NbrStg',NbrStgDef);
options = rdpset(options,'MinNbrStg',MinNbrStgDef);
options = rdpset(options,'MaxNbrStg',MaxNbrStgDef);
options = rdpset(options,'MassFcn',MassDef);


tic
[tout,yout,Stats] = radau(FcnDef,tspan,u0,options,OdeCoeff);    
T_radau = toc

InfoStat = Stats.Stat;
InfoDyn  = Stats.Dyn;


Ue = 0.4*sin(200*pi*tout);
figure(1)
plot(tout,Ue,'b')
hold on
plot(tout,yout(:,5),'r')
grid on
legend('input ','output ',2)
title('Amplifier I')

if InfoDynDef
  figure    
  subplot(4,1,1)
  plot(tout,real(yout(:,1)))   
  hold on
  plot(tout,real(yout(:,1)),'.b')
  grid on
  plot(tout,real(yout(:,5)))
  plot(tout,real(yout(:,5)),'.r')
  title('Amplifier I, Uin  in blue   Uout in red')
    
  subplot(4,1,2)
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept);
  hold on
  grid on
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept,'.k');
  plot(InfoDyn.hreject_t,InfoDyn.hreject,'xr')
  xlim([Ti Tf])
  title('Length of steps in function of time')
    
  subplot(4,1,3)
  plot(InfoDyn.Newt_t,InfoDyn.NewtNbr)
  grid on
  xlim([Ti Tf])
  title('Newton iteration number in function of time')
    
  subplot(4,1,4)
  plot(InfoDyn.Newt_Step,InfoDyn.NewtNbr)
  hold on
  plot(InfoDyn.Jac_Step,0.5,'.b');
  title('Newton iteration number in function of steps and jacobian')
    
  figure
  subplot(2,1,1)
  plot(InfoDyn.NbrStg_t,InfoDyn.NbrStg,'or')
  grid on
  title('Number of Stages versus Time')
  subplot(2,1,2)
  plot(InfoDyn.NbrStg_Step,InfoDyn.NbrStg,'ob')
  grid on
  title('Number of Stages versus Steps Number')
  Nbr1 = length(find(InfoDyn.NbrStg == 1));
  Nbr3 = length(find(InfoDyn.NbrStg == 3));
  Nbr5 = length(find(InfoDyn.NbrStg == 5));
	Nbr7 = length(find(InfoDyn.NbrStg == 7));
  NbrStg_1_3_5_7 = [Nbr1, Nbr3, Nbr5, Nbr7]
end
if InfoStatDef  
  InfoStat = InfoStat
end
 






