% Driver for Ex10 problem.
%  This is a mechanics problem studied in section 4 of Chapter XII of
%  R. Mattheij and J. Molenaar, Ordinary Differential Equations in Theory
%  and Practice, Wiley, New York, 1996.  It describes the motion of a bead 
%  on a rotating needle subject to the forces of gravity, friction, and 
%  centrifugal force.  The equations of motion are
%
%    u1' =   u2
%    u2' = - fr*u2 + lambda*psi(t)
%    u3' =   u4
%    u4' = - fr*u4 - lambda + g/omega^2
%
%  There is a kinematic constraint that corresponds to the bead having
%  to move along the needle:  0 = u3 - u1*psi(t)
%
%  Here the coordinates of the bead are (u1,u3).  The quantity lambda
%  is a Lagrange multiplier.  The function psi(t) = tan(t + phi0), 
%  where phi0 is the initial angle between the needle and the horizontal
%  that is taken here to be pi/4.  The needle is rotating with a constant 
%  angular velocity omega and g is the acceleration due to gravity.  Here
%  this ratio is taken to be 1.  fr is a non-dimensional friction parameter
%  that is taken here to be 10.  The bead is started from (u1,u3) = (1,1) 
%  with initial derivatives (u2,u4) = (-6,-6).
%
%  It is inconvenient that psi becomes infinite during the integration,
%  so we introduce the variable u5 = lambda/cos(t+phi0) in the equations
%  of motion and rewrite the kinematic constraint to obtain the DAE
%
%    u1' =   u2
%    u2' = - 10*u2 + u5*sin(t+pi/4)
%    u3' =   u4
%    u4' = - 10*u4 - u5*cos(t+pi/4) + 1
%     0  =   cos(t+pi/4)*u3 - sin(t+pi/4)*u1 = g(t)
%
%  Replacing the kinematic constraint 0 = g(t) by the implicit constraint
%  0 = g''(t) results in a DAE of index 1.  Unfortunately, as is common with
%  mechanics problems, when this is done the numerical solution drifts so 
%  that the constraint 0 = g(t) is not satisfied well and the computation 
%  breaks down. Here Baumgarte stabilization is done as explained on p. 470 
%  of E. Hairer and G. Wanner, Solving Ordinary Differential Equations II
%  Stiff and Differential-Algebraic Problems, 2nd ed., Springer, Berlin,
%  1996. In this approach the equation is replaced not by 0 = g''(t) but by
%
%    0 = g''(t) + 2*alpha*g'(t) + alpha^2 *g(t)
%
%  for a suitable value of alpha > 0.  Here alpha is taken to be 10 and a 
%  little calculation shows that
%
%    g'(t)  =   cos(t+pi/4)*(u4 - u1) + sin(t+pi/4)*(- u2 - u3)
%    g''(t) =   cos(t+pi/4)*(u4' - u1' - u2 - u3) 
%             + sin(t+pi/4)*(- u2' - u3' - u4 + u1)
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%    
% See also http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
% ---------------------------

clc
clear all
close all
 
addpath D:\RDPSolver

ProblemName = 'Ex11';

% Function and jacobian
FcnDef  = @Ex11Fcn;
MassDef = @Ex11Mass;

% Initial conditions
u0 = zeros(5,1);
u0(1) =  1;
u0(2) = -6;
u0(3) =  1;
u0(4) = -6;
u0(5) =  -10.606601717798210; % Consistent value for the algebraic equation.

Ti    = 0;
Tf    = 15;
tspan = [Ti Tf];

RelTolDef  = 1e-3;
AbsTolDef  = 1e-6;

InfoStatDef = true;
InfoDynDef  = true;

NbrStgDef    = 3; 
MinNbrStgDef = 3;
MaxNbrStgDef = 7;

options = [];
options = rdpset(options,'RelTol',RelTolDef);
options = rdpset(options,'AbsTol',AbsTolDef);
options = rdpset(options,'NbrStg',NbrStgDef);
options = rdpset(options,'MinNbrStg',MinNbrStgDef);
options = rdpset(options,'MaxNbrStg',MaxNbrStgDef);
options = rdpset(options,'MassFcn',MassDef);

tic
[t,u,Stats] = radau(FcnDef,tspan,u0,options);    
T_radau = toc

InfoStat = Stats.Stat;
InfoDyn  = Stats.Dyn;

plot(u(1,1),u(1,3),'o',u(:,1),u(:,3))
title('Path of a bead on a rotating needle.')

figure 
plot(u(:,5))

if InfoDynDef
  figure
  tout = t;
  yout = u;
  subplot(4,1,1)
  plot(tout,real(yout(:,1)))   
  hold on
  plot(tout,real(yout(:,1)),'.b')
  grid on
  plot(tout,real(yout(:,1)))
  plot(tout,real(yout(:,1)),'.r')
  title('Ex11')
    
  subplot(4,1,2)
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept);
  hold on
  grid on
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept,'.k');
  plot(InfoDyn.hreject_t,InfoDyn.hreject,'xr')
  xlim([Ti Tf])
  title('Length of steps in function of time')
    
  subplot(4,1,3)
  plot(InfoDyn.Newt_t,InfoDyn.NewtNbr)
  grid on
  xlim([Ti Tf])
  title('Newton iteration number in function of time')
    
  subplot(4,1,4)
  plot(InfoDyn.Newt_Step,InfoDyn.NewtNbr)
  hold on
  plot(InfoDyn.Jac_Step,0.5,'.b');
  title('Newton iteration number in function of steps and jacobian')
    
  figure
  subplot(2,1,1)
  plot(InfoDyn.NbrStg_t,InfoDyn.NbrStg,'or')
  grid on
  title('Number of Stages versus Time')
  subplot(2,1,2)
  plot(InfoDyn.NbrStg_Step,InfoDyn.NbrStg,'ob')
  grid on
  title('Number of Stages versus Steps Number')
  Nbr1 = length(find(InfoDyn.NbrStg == 1));
  Nbr3 = length(find(InfoDyn.NbrStg == 3));
  Nbr5 = length(find(InfoDyn.NbrStg == 5));
	Nbr7 = length(find(InfoDyn.NbrStg == 7));
  NbrStg_1_3_5_7 = [Nbr1, Nbr3, Nbr5, Nbr7]
end
if InfoStatDef  
  InfoStat = InfoStat
end
  





