% Driver for Ex12 problem.
%  This is Problem 1 from P.M. Dew and J.E. Walsh, A set of library 
%  routines for solving parabolic equations in one space variable, ACM
%  Trans. Math. Software, 7(1981) 295-314.  They use it to illustrate
%  the capability of their method of lines code for solving problems
%  of both elliptic and parabolic type.  In this approach to solving
%  PDEs, the parabolic equations give rise to ODEs and the elliptic 
%  equations to algebraic equations.  The two equations are
%
%           0 = D^2(u1)/Dx^2 - exp(-2*u2)
%    D(u2)/Dt = D^2(u2)/Dx^2 + exp(-2*u2) - exp(-u1) 
%
%  The problem is set on 0 <= x <= 1 and t >= 0.  The initial values are  
%  u2(x,0) = log(x+p)  for a parameter p > 1.  Here we take p = 10.
%
%  The boundary conditions at x = 0 are D(u1)/Dx = exp(-u1) and 
%  u2 = log(p+t).  The boundary conditions at x = 1 are u1 = log(1+p+t)
%  and D(u2)/Dx = exp(-u1).
%
%  The analytical solution is u1 = u2 = log(x+p+t).
%  
%  This example is interesting because it is representative of a class
%  of DAEs, it involves a "large" number of algebraic equations, and the
%  obvious initial conditions are inconsistent. The problem is solved
%  quickly, but it is sufficiently large and the Jacobian is sufficiently
%  sparse that it is a good idea to inform the code about the structure
%  of the Jacobian.  This is done to illustrate how to proceed.
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%    
% See also http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
% ---------------------------

clc
clear all
close all
 
addpath D:\RDPSolver

ProblemName = 'Ex12';

% Function and jacobian
FcnDef    = @Ex12Fcn;
MassDef   = @Ex12Mass;

P = 10;
N = 20;

OdeCoeff(1) = P;
OdeCoeff(2) = N;

%  Use a mesh of N equally spaced points in 0 <= x <= 1.
x = linspace(0,1,N)';
%  Initial conditions are taken from the analytical solution.  
%  NOTE that they are not consistent for the DAE resulting from 
%  discretization, except for the two Dirichlet boundary conditions.
u10 = log(x + P);
u20 = u10;
y0  = [u10; u20];

% Consistent initial conditions:
y0 = [  2.301708858125273   2.307004328613576   2.312272387572591 ...
        2.317513319795494   2.322727405660106   2.327914921219691 ...
        2.333076138291438   2.338211324542694   2.343320743575007 ...
        2.348404655006054   2.353463314549522   2.358496974092990 ...
        2.363505881773881   2.368490282053537   2.373450415789476 ...
        2.378386520305885   2.383298829462389   2.388187573721164 ...
        2.393052980212430   2.397895272798371   2.302585092994046 ...
        2.307834448880189   2.313056392861341   2.318251209738445 ...
        2.323419179896888   2.328560579397307   2.333675680064077 ...
        2.338764749571548   2.343828051528095   2.348865845558052 ...
        2.353878387381596   2.358865928892636   2.363828718234764 ...
        2.368766999875347   2.373681014677776   2.378570999971968 ...
        2.383437189623141   2.388279814098929   2.393099100534878 ...
        2.397895272798371]';
      
Ti    = 0;
Tf    = 1;
dt    = 0.5;
tspan = [0 0.5 1];

RelTolDef  = 1e-6;
AbsTolDef  = 1e-6;

InfoStatDef = true;
InfoDynDef  = true;

NbrStgDef    = 3; 
MinNbrStgDef = 3;
MaxNbrStgDef = 7;

options = [];
options = rdpset(options,'RelTol',RelTolDef);
options = rdpset(options,'AbsTol',AbsTolDef);
options = rdpset(options,'NbrStg',NbrStgDef);
options = rdpset(options,'MinNbrStg',MinNbrStgDef);
options = rdpset(options,'MaxNbrStg',MaxNbrStgDef);
options = rdpset(options,'MassFcn',MassDef);
% options = rdpset(options,'JacPatFcn',JacPatDef);

tic
[t,y,Stats] = radau(FcnDef,tspan,y0,options,OdeCoeff);    
T_radau = toc

InfoStat = Stats.Stat;
InfoDyn  = Stats.Dyn;

u1 = y(:,1:N);
u2 = y(:,1+N:N+N);

Exact = [log(x'+P+0); log(x'+P+0.5); log(x'+P+1)];
plot(x,u1,'*',x,u2,'o',x,Exact)
xlabel('u1 = *, u2 = o, solid line is the analytical solution')
title('MOL Solution of Elliptic-Parabolic System of PDEs')

if InfoDynDef
  figure
  tout = t;
  yout = u1;
  subplot(4,1,1)
  plot(tout,real(yout(:,1)))   
  hold on
  plot(tout,real(yout(:,1)),'.b')
  grid on
  plot(tout,real(yout(:,1)))
  plot(tout,real(yout(:,1)),'.r')
  title('Ex12')
    
  subplot(4,1,2)
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept);
  hold on
  grid on
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept,'.k');
  plot(InfoDyn.hreject_t,InfoDyn.hreject,'xr')
  xlim([Ti Tf])
  title('Length of steps in function of time')
    
  subplot(4,1,3)
  plot(InfoDyn.Newt_t,InfoDyn.NewtNbr)
  grid on
  xlim([Ti Tf])
  title('Newton iteration number in function of time')
    
  subplot(4,1,4)
  plot(InfoDyn.Newt_Step,InfoDyn.NewtNbr)
  hold on
  plot(InfoDyn.Jac_Step,0.5,'.b');
  title('Newton iteration number in function of steps and jacobian')
   
  figure
  subplot(2,1,1)
  plot(InfoDyn.NbrStg_t,InfoDyn.NbrStg,'or')
  grid on
  title('Number of Stages versus Time')
  subplot(2,1,2)
  plot(InfoDyn.NbrStg_Step,InfoDyn.NbrStg,'ob')
  grid on
  title('Number of Stages versus Steps Number')
  Nbr1 = length(find(InfoDyn.NbrStg == 1));
  Nbr3 = length(find(InfoDyn.NbrStg == 3));
  Nbr5 = length(find(InfoDyn.NbrStg == 5));
	Nbr7 = length(find(InfoDyn.NbrStg == 7));
  NbrStg_1_3_5_7 = [Nbr1, Nbr3, Nbr5, Nbr7]
end
if InfoStatDef  
  InfoStat = InfoStat
end

