function ex12
%  This is Problem 1 from P.M. Dew and J.E. Walsh, A set of library 
%  routines for solving parabolic equations in one space variable, ACM
%  Trans. Math. Software, 7(1981) 295-314.  They use it to illustrate
%  the capability of their method of lines code for solving problems
%  of both elliptic and parabolic type.  In this approach to solving
%  PDEs, the parabolic equations give rise to ODEs and the elliptic 
%  equations to algebraic equations.  The two equations are
%
%           0 = D^2(u1)/Dx^2 - exp(-2*u2)
%    D(u2)/Dt = D^2(u2)/Dx^2 + exp(-2*u2) - exp(-u1) 
%
%  The problem is set on 0 <= x <= 1 and t >= 0.  The initial values are  
%  u2(x,0) = log(x+p)  for a parameter p > 1.  Here we take p = 10.
%
%  The boundary conditions are x = 0 are D(u1)/Dx = exp(-u1) and 
%  u2 = log(p+t).  The boundary conditions at x = 1 are u1 = log(1+p+t)
%  and D(u2)/Dx = exp(-u1).
%
%  The analytical solution is u1 = u2 = log(x+p+t).
%  
%  This example is interesting because it is representative of a class
%  of DAEs, it involves a "large" number of algebraic equations, and the
%  obvious initial conditions are inconsistent. The problem is solved
%  quickly, but it is sufficiently large and the Jacobian is sufficiently
%  sparse that it is a good idea to inform the code about the structure
%  of the Jacobian.  This is done to illustrate how to proceed.

% Copyright 2004, The MathWorks, Inc.

% Problem parameters -- shared with the nested function
P = 10;
N = 20;

%  Use a mesh of N equally spaced points in 0 <= x <= 1.
x = linspace(0,1,N)';

%  Initial conditions are taken from the analytical solution.  
%  NOTE that they are not consistent for the DAE resulting from 
%  discretization, except for the two Dirichlet boundary conditions.

u10 = log(x + P);
u20 = u10;
y0 = [u10; u20];

M = diag([zeros(N+1,1); ones(N-1,1)]);
options = odeset('Mass',M,'JPattern',Jacpat(N));

tic
[t,y] = ode15s(@ex12ode,[0 0.5 1],y0,options);
T_ode15s = toc
 
u1    = y(:,1:N);
u2    = y(:,1+N:N+N);
exact = [log(x'+P+0); log(x'+P+0.5); log(x'+P+1)];
figure
plot(x,u1,'*',x,u2,'o',x,exact)
xlabel('u1 = *, u2 = o, solid line is the analytical solution')
title('MOL Solution of Elliptic-Parabolic System of PDEs')


%------------------------------------------------------------------------------
% Nested function -- parameters shared with the outer function.

  function yp = ex12ode(t,y)
  %  Evaluate the differential equations.

    u1 = y(1:N);
    u2 = y(1+N:N+N);
    f1 = zeros(N,1);
    f2 = zeros(N,1);
    
    %  The are N equally spaced mesh points in 0 <= x <= 1:
    h = 1/(N - 1);
    
    %  Equations at interior mesh points:
    for i = 2:(N-1)
      f1(i) = (u1(i+1) - 2*u1(i) + u1(i-1))/h^2 + exp(-2*u2(i));              
      f2(i) = (u2(i+1) - 2*u2(i) + u2(i-1))/h^2 + exp(-2*u2(i)) + exp(-u1(i));
    end
    
    %  Boundary conditions at x = 0:
    f1(1) = ((u1(2) - u1(1))/h - exp(-u1(1)))/h - exp(-2*u2(1));
    f2(1) = u2(1) - log(P + t);
    
    %  Boundary conditions at x = 1:  
    f1(N) = u1(N) - log(1 + P + t);
    f2(N) = (exp(-u1(N)) - (u2(N) - u2(N-1))/h)/h + exp(-2*u2(N)) + exp(-u1(N));
    
    yp = [f1; f2];
    
  end  % function ex12ode

%------------------------------------------------------------------------------

end  % ex12

%------------------------------------------------------------------------------
% Subfunction

function S = Jacpat(N)
%  Define the sparsity structure of the Jacobian by indicating which 
%  variables appear in which equations.  This is done by inspection of
%  the function 'f' above for evaluating the equations.

S = sparse(2*N,2*N);
S(1,1) = 1;          %  u1(1)
S(1,2) = 1;          %  u1(2)
S(1,1+N) = 1;        %  u2(1)
for i = 2:(N-1)
  S(i,i-1) = 1;      %  u1(i-1)
  S(i,i) = 1;        %  u1(i)
  S(i,i+1) = 1;      %  u1(i+1) 
  S(i,i+N) = 1;      %  u2(i)
end
S(N,N) = 1;          %  u1(N)  
S(1+N,1+N) = 1;      %  u2(1)
for i = 2:(N-1)
  S(i+N,i) = 1;      %  u1(i)
  S(i+N,i-1+N) = 1;  %  u2(i-1)
  S(i+N,i+N) = 1;    %  u2(i)
  S(i+N,i+1+N) = 1;  %  u2(i+1)
end
S(N+N,N) = 1;        %  u1(N)
S(N+N,N-1+N) = 1;    %  u2(N-1)
S(N+N,N+N) = 1;      %  u2(N)  

end  % function Jacpat

