function yp = Ex13Fcn(t,y)
%  This is the discharge pressure control problem from E. Hairer, Ch. Lubich,
%  and M. Roche, The Numerical Solution of Differential-Algebraic Systems by
%  Runge-Kutta Methods, Lecture Notes in Math. #1409, Springer, Berlin, 1989.
%  As posed by Preston, Berzins, Dew, and Scales, HLR show that the problem
%  is of index 2.  They do this by reducing it to a problem that is shown
%  to be of index 1.  The index 1 problem is solved here.  The variables of
%  their analysis correspond to the ones used here as follows:
%
%    k = y(1), c = y(2), M = y(3), p = y(4), P = y(5), m = y(6).
%
%  For analytical purposes HLR introduce a variable s = c + p/15.  This is
%  not used here with the consequence that the mass matrix is not diagonal,
%  hence the problem is not in semi-explicit form.  HLR solve the problem
%  with RADAU5, which requires consistent initial conditions.  The values
%  given the code here are inconsistent values that approximate well their
%  consistent values.  The consistent initial conditions computed by the
%  code are close to the values of HLR, but they are not identical.  The
%  plot of the given flow in and the computed flow out computed here agrees 
%  with Fig. 9.8 of HLR.
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%    
% See also http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
% ---------------------------

%  k = y(1), c = y(2), M = y(3), p = y(4), P = y(5), m = y(6)

mu  = 15 + 5*tanh(t - 10);
mup = 5/cosh(t - 10)^2;

yp    = zeros(6,1);
yp(1) =  (y(2) - y(1))/20;
yp(2) = -(y(4) - 99.1)/75;
yp(3) =  mu - y(6);
yp(4) =  (3.35 - 0.075*y(6) + 0.001*y(6)^2) - y(4)/y(5);
yp(5) =  20*y(5) - y(3);
yp(6) =  y(3)*yp(3)/400 + mu*mup/(1.2*y(1))^2 ...
           - mu^2 * yp(1)/(1.2^2 * y(1)^3); 