% Driver for the Ex13 problem.
%  This is the discharge pressure control problem from E. Hairer, Ch. Lubich,
%  and M. Roche, The Numerical Solution of Differential-Algebraic Systems by
%  Runge-Kutta Methods, Lecture Notes in Math. #1409, Springer, Berlin, 1989.
%  As posed by Preston, Berzins, Dew, and Scales, HLR show that the problem
%  is of index 2.  They do this by reducing it to a problem that is shown
%  to be of index 1.  The index 1 problem is solved here.  The variables of
%  their analysis correspond to the ones used here as follows:
%
%    k = y(1), c = y(2), M = y(3), p = y(4), P = y(5), m = y(6).
%
%  For analytical purposes HLR introduce a variable s = c + p/15.  This is
%  not used here with the consequence that the mass matrix is not diagonal,
%  hence the problem is not in semi-explicit form.  HLR solve the problem
%  with RADAU5, which requires consistent initial conditions.  The values
%  given the code here are inconsistent values that approximate well their
%  consistent values.  The consistent initial conditions computed by the
%  code are close to the values of HLR, but they are not identical.  The
%  plot of the given flow in and the computed flow out computed here agrees 
%  with Fig. 9.8 of HLR.
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%    
% See also http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
% ---------------------------

clc
clear all
close all
 
addpath D:\RDPSolver

ProblemName = 'Ex13';

% Function and jacobian
FcnDef  = @Ex13Fcn;
MassDef = @Ex13Mass;

% Parameter
mu = 15 + 5*tanh(0 - 10);


%  Consistent initial values (up to 1e-15)
y0    = zeros(6,1);
y0(1) = 0.25;
y0(2) = 0.25;
y0(3) = 734;
y0(4) = 99.0899949200203;
y0(5) = 36.7;
y0(6) = 10.00000251671038;

% Consitency :
% yp(4:6) = 1.0e-014 * [ 0.088817841970013,0,-0.183914654474726];
 
Ti    = 0;
Tf    = 40;
tspan = [0  40];

RelTolDef  = 1e-3;
AbsTolDef  = 1e-6;

InfoStatDef = true;
InfoDynDef  = true;

NbrStgDef    = 5; 
MinNbrStgDef = 3;
MaxNbrStgDef = 7;

RefineDef    = 1;

options = [];
options = rdpset(options,'RelTol',RelTolDef);
options = rdpset(options,'AbsTol',AbsTolDef);
options = rdpset(options,'NbrStg',NbrStgDef);
options = rdpset(options,'MinNbrStg',MinNbrStgDef);
options = rdpset(options,'MaxNbrStg',MaxNbrStgDef);
options = rdpset(options,'MassFcn',MassDef);
options = rdpset(options,'Refine',RefineDef);

tic
[t,y,Stats] = radau(FcnDef,tspan,y0,options);    
T_radau = toc

InfoStat = Stats.Stat;
InfoDyn  = Stats.Dyn;

mu = 15 + 5*tanh(t - 10);
plot(t,mu,t,y(:,6))
legend('given flow in ','computed flow out ')
axis([0 40 0 40])
title('Discharge Pressure Control Problem')
grid on

if InfoDynDef
  figure
  tout = t;
  yout = y;
  subplot(4,1,1)
  plot(tout,real(yout(:,1)))   
  hold on
  plot(tout,real(yout(:,1)),'.b')
  grid on
  plot(tout,real(yout(:,1)))
  plot(tout,real(yout(:,1)),'.r')
  title('Ex13')
    
  subplot(4,1,2)
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept);
  hold on
  grid on
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept,'.k');
  plot(InfoDyn.hreject_t,InfoDyn.hreject,'xr')
  xlim([Ti Tf])
  title('Length of steps in function of time')
    
  subplot(4,1,3)
  plot(InfoDyn.Newt_t,InfoDyn.NewtNbr)
  grid on
  xlim([Ti Tf])
  title('Newton iteration number in function of time')
    
  subplot(4,1,4)
  plot(InfoDyn.Newt_Step,InfoDyn.NewtNbr)
  hold on
  plot(InfoDyn.Jac_Step,0.5,'.b');
  title('Newton iteration number in function of steps and jacobian')
    
  figure
  subplot(2,1,1)
  plot(InfoDyn.NbrStg_t,InfoDyn.NbrStg,'or')
  grid on
  title('Number of Stages versus Time')
  subplot(2,1,2)
  plot(InfoDyn.NbrStg_Step,InfoDyn.NbrStg,'ob')
  grid on
  title('Number of Stages versus Steps Number')
  Nbr1 = length(find(InfoDyn.NbrStg == 1));
  Nbr3 = length(find(InfoDyn.NbrStg == 3));
  Nbr5 = length(find(InfoDyn.NbrStg == 5));
  Nbr7 = length(find(InfoDyn.NbrStg == 7));
  NbrStg_1_3_5_7 = [Nbr1, Nbr3, Nbr5, Nbr7]
end
if InfoStatDef  
  InfoStat = InfoStat
end
 





