% Driver for the DAE1 simple model
% Differential equations:
%   epsilon*y'' + 2*y' + y = 0
% or
%           y' = z
%   epsilon*z' = -y - 2*z
% The epsilon value is given in the mass function
%
% If epsilon == 0, the initial conditions [y0,z0] must be [y0,-0.5y0] 
% (consistent conditions) and the solution is :
%    y = y0*exp(-0.5*t)
% if epsilon ~= 0 the solution is :
%    y = y1*exp(L1*t) + y2*exp(L2*t)
% with 
% L1 = (-1+sqrt(1-epsilon))/(epsilon)
% L2 = (-1-sqrt(1-epsilon))/(epsilon)
% Let the initial conditions [y0,-0.5y0], the solution is
% y(t) = y0 * ((L2-0.5)/(L2-L1)exp(L1*t) + (L1-0.5)/(L1-L2) )
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%    
% See also  http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
% ---------------------------

% Options:  DAE index 1
% Input:    DAE1Fcn DAE1Jac DAE1Mass epsilon
% Output:   [tout, yout, InfoStat, InfoDyn]
clc
close all
clear all

addpath D:\RDPSolver

ProblemName = 'DAE1';

% Function and jacobian
FcnDef  = @DAE1Fcn;
JacDef  = @DAE1Jac;
MassDef = @DAE1Mass;

Ti    = 0;
Tf    = 20;
tspan = [Ti Tf];

% Initial conditions
Y0 = 1; Z0 = -0.5*Y0;
y0 = [Y0;Z0];

InfoStatDef = true;
InfoDynDef  = true;

% Parameters
epsilonvec(1) = 0.9;
epsilonvec(2) = 1e-5;
epsilonvec(3) = 0;

tol        = 1e-6;
RelTolDef  = tol;
AbsTolDef  = tol;

NbrStgDef    = 3; 
MinNbrStgDef = 3;
MaxNbrStgDef = 7;

options = [];
options = rdpset(options,'RelTol',RelTolDef);
options = rdpset(options,'AbsTol',AbsTolDef);
options = rdpset(options,'NbrStg',NbrStgDef);
options = rdpset(options,'MinNbrStg',MinNbrStgDef);
options = rdpset(options,'MaxNbrStg',MaxNbrStgDef);
options = rdpset(options,'JacFcn',JacDef);
options = rdpset(options,'MassFcn',MassDef);

for k = 1: length(epsilonvec)
  epsilon = epsilonvec(k);
  tic
  [tout,yout] = radau(FcnDef,tspan,y0,options,epsilon);    
  T_radau = toc

  if epsilon == 0
    yexact = Y0*exp(-0.5*tout);
  else  
    if epsilon < 1e-4
      L1 = - 0.5 - epsilon/8 - 1/16*epsilon^2 - 5/(24*16)*epsilon^3 ...
           - 1/(120*35/32)*epsilon^4;     
      L2 = (-1-sqrt(1-epsilon))/epsilon;
    else
      L1 = (-1+sqrt(1-epsilon))/epsilon;
      L2 = (-1-sqrt(1-epsilon))/epsilon;
    end
    yexact = Y0*( (L2+0.5)/(L2-L1) * exp(L1*tout) + ...
                  (L1+0.5)/(L1-L2) * exp(L2*tout) );              
  end

  figure(k)
  subplot(2,1,1)
  plot(tout,real(yout(:,1)))
  hold on
  plot(tout,real(yout(:,1)),'.r')
  grid on
  title(['Solution of:  epsilon*y" +2*y''+y = 0,  ','epsilon = ', num2str(epsilon)])
  subplot(2,1,2)
  plot(tout,real(yout(:,1)) - yexact)
  grid on
  title('Error, numerical solution  minus  exact solution')
  
end
