% Driver for the Pendulum problem
%  The pendulum problem arises most naturally as an index 3 problem
%  with an algebraic equation that says the length L of the pendulum
%  is constant.  It can be reduced by differentiation to an index 2
%  problem and by another to an index 1 problem.  
%  In this example, the pendulum problem is solved as an index 1, index 2
%  and index 3 DAE.
%
%  The differential equations are
%        y(1)' =  y(3)            or   p' = u                            
%        y(2)' =  y(4)            or   q' = v                                 
%      m*y(3)' = -y(1)*y(5)     or  mu' = -p*lambda                       
%      m*y(4)' = -y(2)*y(5)     or  mv' = -q*lambda - g   
%
%  The algebraic equation for index 3 is                   
%        0 = y(1)^2 + y(2)^2      or  0  = p^2 + q^2 - L^2 
%
%  The algebraic equation for index 2 is
%        0 = y(1)*y(3) + y(2)*y(4) or  0  = p*u + q*v   
%
%  The algebraic equation for index 1 is
%        0  = m*(y(3)^2 + y(4)^2) -g*y(2) - y(5)*L^2  or
%        0  = m*(u^2 + v^2) - g*q - lambda*L^2
%
%   where  y(1) = p, 
%          y(2) = q, 
%          y(3) = u, 
%          y(4) = v, 
%          y(5) = lambda
%
%   Parameters  m = 1, L = 1, g = 1.
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%    
% See also  http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
% ---------------------------

% Options:  DAE index 1,2,3
% Input:    PendulumFcn,PendulumJac,PendulumMass, Index, Parameters
%           Tolerance, NbrStg
% Output:   [tout, yout, InfoStat, InfoDyn]

clc
clear all
close all
 
addpath D:\RDPSolver

ProblemName = 'Pendulum';

% Function and jacobian
FcnDef  = @PendulumFcn;
MassDef = @PendulumMass;
JacDef  = @PendulumJac;

Ti    = 0;
Tf    = 10.0;
tspan = [Ti Tf];

InfoStatDef = true;
InfoDynDef  = true;

RelTolDef  = 1e-6;  % Important: high RelTol
AbsTolDef  = 1e-3;  % Possible low AbsTol

NbrStgDef    = 5;  % Important for Index == 3 
MinNbrStgDef = 5;  % MinNbrStgDef >= 5
MaxNbrStgDef = 7;

options = [];
options = rdpset(options,'RelTol',RelTolDef);
options = rdpset(options,'AbsTol',AbsTolDef);
options = rdpset(options,'NbrStg',NbrStgDef);
options = rdpset(options,'MinNbrStg',MinNbrStgDef);
options = rdpset(options,'MaxNbrStg',MaxNbrStgDef);
options = rdpset(options,'MassFcn',MassDef);
options = rdpset(options,'JacFcn',JacDef);

for Index = 1:3

  if Index == 1    
    % Initial conditions
    y0         = zeros(5,1);
    y0(1)      = 1;
    NbrInd1Def = 5;
    options = rdpset(options,'NbrInd1',NbrInd1Def);
    tic
    [t,y] = radau(FcnDef,tspan,y0,options,Index); 
    T_radau = toc
    figure(Index)
    plot(t,y(:,1:4))
    title('Pendulum Problem as Index 1 DAE')
    grid on
  elseif Index == 2
    % Initial conditions
    y0         = zeros(5,1);
    y0(1)      = 1;
    NbrInd1Def = 4;
    NbrInd2Def = 1;
    options = rdpset(options,'NbrInd1',NbrInd1Def);
    options = rdpset(options,'NbrInd2',NbrInd2Def);
    tic
    [t,y] = radau(FcnDef,tspan,y0,options,Index); 
    T_radau = toc
    figure(Index)
    plot(t,y(:,1:4))
    title('Pendulum Problem as Index 2 DAE')
    grid on
  elseif Index == 3
    y0         = zeros(5,1);
    y0(1)      = 1;
    NbrInd1Def = 4;
    NbrInd2Def = 0;
    NbrInd3Def = 1;
    options = rdpset(options,'NbrInd1',NbrInd1Def);
    options = rdpset(options,'NbrInd2',NbrInd2Def);
    options = rdpset(options,'NbrInd3',NbrInd3Def);
    tic
    [t,y] = radau(FcnDef,tspan,y0,options,Index); 
    T_radau = toc
    figure(Index)
    plot(t,y(:,1:4))
    title('Pendulum Problem as Index 3 DAE')
    grid on
  end
end




