% Driver for the Van der Pol problem
% Differential equations:
% dy1/dt = y2
% dy2/dt = ( (1-y1^2)*y2 - y1 )/epsilon
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%    
% See also  http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
% ---------------------------

% Options:  MaxStep
% Input:    (VdP,tspan,y0,[],epsilon)
% Output:   odeplot or [tout, yout, InfoStat, InfoDyn]

clc
close all
clear all

addpath D:\RDPSolver

ProblemName = 'Van der Pol';

% Ode Function
FcnDef = @VdPFcn;

% Parameters
epsilon = 1e-6;

% Initial conditions 
y0 = [2; 0.0]; 

Ti    = 0;
Tf    = 2;
tspan = [Ti Tf];

options    = [];
MaxStepDef = 1e-3;
options    = rdpset(options,'MaxStep',MaxStepDef);

tic
radau(FcnDef,tspan,y0,options,epsilon);
T_radau = toc

tic
[tout,yout] = radau(FcnDef,tspan,y0,options,epsilon);
T_radau = toc

tic
[tout,yout,Stats] = radau(FcnDef,tspan,y0,options,epsilon);
T_radau = toc

InfoStat = Stats.Stat;
InfoDyn  = Stats.Dyn;

figure
plot(tout,real(yout(:,1)))
hold on
plot(tout,real(yout(:,1)),'.r')
grid on


figure  
subplot(4,1,1)
plot(tout,real(yout(:,1)))
grid on
hold on
plot(tout,real(yout(:,1)),'.r')
title('Van der Pol, component no 1')
    
subplot(4,1,2)
semilogy(InfoDyn.haccept_t,InfoDyn.haccept);
hold on
grid on
semilogy(InfoDyn.haccept_t,InfoDyn.haccept,'.k');
plot(InfoDyn.hreject_t,InfoDyn.hreject,'xr')
xlim([Ti Tf])
title('Length of steps in function of time')
    
subplot(4,1,3)
plot(InfoDyn.Newt_t,InfoDyn.NewtNbr)
grid on
xlim([Ti Tf])
title('Newton iteration number in function of time')
    
subplot(4,1,4)
plot(InfoDyn.Newt_Step,InfoDyn.NewtNbr)
hold on
plot(InfoDyn.Jac_Step,0.5,'.b');
title('Newton iteration number in function of steps and jacobian')
    
figure
subplot(2,1,1)
plot(InfoDyn.NbrStg_t,InfoDyn.NbrStg,'or')
grid on
title('Number of Stages versus Time')
subplot(2,1,2)
plot(InfoDyn.NbrStg_Step,InfoDyn.NbrStg,'ob')
grid on
title('Number of Stages versus Steps Number')
Nbr1 = length(find(InfoDyn.NbrStg == 1));
Nbr3 = length(find(InfoDyn.NbrStg == 3));
Nbr5 = length(find(InfoDyn.NbrStg == 5));
Nbr7 = length(find(InfoDyn.NbrStg == 7));
NbrStg_1_3_5_7 = [Nbr1,Nbr3,Nbr5,Nbr7]    

InfoStat = InfoStat
 
% Comparaison numerical solution - "exact" solution
if epsilon == 1e-6 && y0(1) == 2 && y0(2) == 0
  % The following values are y(1), y(2) evaluatetd at time
  % t = 1,2,3 ... 
  yexact(1,:)  = [-0.1863646254808130E+01,  0.7535430865435460E+00];
  yexact(2,:)  = [ 0.1706167732170456E+01, -0.8928097010248257E+00];
  yexact(3,:)  = [-0.1510606936744095E+01,  0.1178380000730945E+01];
  yexact(4,:)  = [ 0.1194414677695905E+01, -0.2799585996540082E+01];
  yexact(5,:)  = [ 0.1890428596416747E+01, -0.7345118680166940E+00];
  yexact(6,:)  = [-0.1737716306805883E+01,  0.8604008653025923E+00];
  yexact(7,:)  = [ 0.1551614645548223E+01, -0.1102382892533321E+01];
  yexact(8,:)  = [-0.1278631984330405E+01,  0.2013890883009155E+01];
  yexact(9,:)  = [-0.1916552949489830E+01,  0.7169573003463228E+00];
  yexact(10,:) = [ 0.1768163792391936E+01, -0.8315276407898496E+00];
  yexact(11,:) = [-0.1590150544829062E+01,  0.1040279389212485E+01];

  y0 = [2; 0.0]; 
  tspan = [0 1];
  for k = 1:11
    [tout,yout] = radau(FcnDef,tspan,y0,options,epsilon);
    [M,N] = size(yout);
    erry(k,:) = yout(M,:) - yexact(k,:);
    tspan = tspan + 1;
    y0    = yout(M,:);
    y0    = y0(:);
  end
  figure
  subplot(2,1,1)
  plot(real(erry(:,1)))
  grid on
  title('Error of the first component')

  subplot(2,1,2)
  plot(real(erry(:,2)))
  grid on  
  title('Error of the second component')
end
