% Driver for the harmonic oscillator
% Differential equations:
% y" + y = 0
% or
% du/dt = v
% dv/dt = -u
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%     
% See also http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
% ---------------------------

clc
close all
clear all

addpath D:\RDPSolver

ProblemName = 'Oscillator';

% Function and jacobian
FcnDef  = @OscFcn;
JacDef  = @OscJac;
MassDef = @OscMass;

% Initial conditions
y0 = [1;0];

Ti    = 0;
Tf    = -40*pi;
tspan = [Ti Tf];

tol        = 1e-6;
RelTolDef  = tol;
AbsTolDef  = tol;

InfoStatDef   = true;
InfoDynDef    = true;

NbrStgDef    = 1; 
MinNbrStgDef = 1;
MaxNbrStgDef = 7;
RefineDef    = 10;

options = [];
options = rdpset(options,'RelTol',RelTolDef);
options = rdpset(options,'AbsTol',AbsTolDef);
options = rdpset(options,'NbrStg',NbrStgDef);
options = rdpset(options,'MinNbrStg',MinNbrStgDef);
options = rdpset(options,'MaxNbrStg',MaxNbrStgDef);
options = rdpset(options,'Refine',RefineDef);
options = rdpset(options,'JacFcn',JacDef);
options = rdpset(options,'MassFcn',MassDef);

tic
[tout,yout,Stats] = radau(FcnDef,tspan,y0,options);    
toc

InfoStat = Stats.Stat;
InfoDyn  = Stats.Dyn;

figure
subplot(2,1,1)
plot(tout,real(yout(:,1)))
hold on
plot(tout,real(yout(:,1)),'.r')
grid on
title('Oscillator, first Component') 
subplot(2,1,2)
plot(tout,real(yout(:,1))- cos(tout))
grid on
title('Oscillator, Error on the first Component')
 
if InfoDynDef
  figure
   
  subplot(4,1,1)
  plot(tout,real(yout(:,1)))
  grid on
  hold on
  plot(tout,real(yout(:,1)),'.r')
  title('Oscillator first Component')
    
  subplot(4,1,2)
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept);
  hold on
  grid on
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept,'.k');
  plot(InfoDyn.hreject_t,InfoDyn.hreject,'xr')
  title('Length of steps in function of time')
    
  subplot(4,1,3)
  plot(InfoDyn.Newt_t,InfoDyn.NewtNbr)
  grid on
  title('Newton iteration number in function of time')
    
  subplot(4,1,4)
  plot(InfoDyn.Newt_Step,InfoDyn.NewtNbr)
  hold on
  plot(InfoDyn.Jac_Step,0.5,'.b');
  title('Newton iteration number in function of steps and Jacobian')
    
  figure
  plot(InfoDyn.NbrStg_t,InfoDyn.NbrStg,'or')
  grid on
  title('Number of Stages versus Time')
  figure
  plot(InfoDyn.NbrStg_Step,InfoDyn.NbrStg,'ob')
  grid on
  title('Number of Stages versus Steps Number')
  Nbr1 = length(find(InfoDyn.NbrStg == 1));
  Nbr3 = length(find(InfoDyn.NbrStg == 3));
  Nbr5 = length(find(InfoDyn.NbrStg == 5));
  Nbr7 = length(find(InfoDyn.NbrStg == 7));
  NbrStg_1_3_5_7 = [Nbr1,Nbr3,Nbr5,Nbr7]
end
if InfoStatDef
  InfoStat = InfoStat
end
  