function yPrime = OscTunnelDiodeFcn(t,y,OdeCoeff)
% Model of the circuit
% Tunnel diode resistance self and condensator in serie with a
% constant voltage source.
% Tunnel diode model:
% In the tunnel diode, the voltage-current looks like:
%   Id = Ip *(Ud/Up) .* exp(1.0 - Ud/Up) + ...
%        Iv*exp(-A2*Uv)*(exp(A2*Ud)-1)  + ...
%        Is * (exp(Ud/Ut) - 1.0);
%   Id : current in the diode
%   Ud : diode voltage
%   Ip : peak current
%   Iv : valley current
%   Up : peak voltage
%   Uv : valley voltage
%   Equations (Kirchhoff)
%   L*dI/dt + Ud + Uc = Ug
%   I = C*Uc' 
%   L*CUd'' + R(C*Ud')*C*Ud' + Ud = Ug
%   or
%   Ud'' = 1/(LC)*(Ug-Ud-R(C*Ud')*C*Ud' )
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%     
% See also http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
% ---------------------------
Ut       = OdeCoeff{1};
Up       = OdeCoeff{2};
Ip       = OdeCoeff{3};
Uv       = OdeCoeff{4};
Iv       = OdeCoeff{5};
Is       = OdeCoeff{6};
A2       = OdeCoeff{7};
L        = OdeCoeff{8};
C        = OdeCoeff{9};
R        = OdeCoeff{10};
Ug0      = OdeCoeff{11};

Ud       = y(1);
[Id,Id1,Id2] = TunnelDiode_Ir_dIr_d2Ir(Ud,OdeCoeff);

Borne = 2e-5;
Denom = L*Id1;
if abs(Denom) < Borne
  Denom = Borne * sign(Denom);
end
if Denom == 0;
  Denom = Borne;
end

yPrime(1) = y(2);
yPrime(2) = ((Ug0-Ud)/(R*C)-L*Id2*y(2).^2 - (1+L/(R*C)*Id1)*y(2) - Id/C)/Denom;
yPrime    = yPrime(:);
