% Driver for the tunnel diode oscillator problem
% Model of the circuit
% Tunnel diode resistance self and condensator in serie with a
% constant voltage source.
% Tunnel diode model:
% In the tunnel diode, the voltage-current looks like:
%   Id = Ip *(Ud/Up) .* exp(1.0 - Ud/Up) + ...
%        Iv*exp(-A2*Uv)*(exp(A2*Ud)-1)  + ...
%        Is * (exp(Ud/Ut) - 1.0);
%   Id : current in the diode
%   Ud : diode voltage
%   Ip : peak current
%   Iv : valley current
%   Up : peak voltage
%   Uv : valley voltage
%   Equations (Kirchhoff)
%   L*dI/dt + Ud + Uc = Ug
%   I = C*Uc' 
%   L*CUd'' + R(C*Ud')*C*Ud' + Ud = Ug
%   or
%   Ud'' = 1/(LC)*(Ug-Ud-R(C*Ud')*C*Ud' )
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%     
% See also http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
% ---------------------------

clc
close all
clear all

warning on

addpath D:\RDPSolver

ProblemName = 'Tunnel diode oscillator';

% Set this to 1 if you want graphical output via 'odeplot'
graphics = 0;

% Function and jacobian
FcnDef = @OscTunnelDiodeFcn;
JacDef = [];

% Parameters
Ut = 25.84e-3;    %23.8e-3 ;    
Up = 0.065; 
Ip = 4.7e-3; 
Uv = 0.355; 
Iv = 0.45e-3;      %0.372e-3;
Is = 50e-12;       %25e-12; 
A2 = 15;

L   = 1e-7;
C   = 1e-7;
Ug0 = 0.2;    

% Initial conditions 
y0  = [0.001;0.0]; 

OdeCoeff{1}  = Ut;
OdeCoeff{2}  = Up ;
OdeCoeff{3}  = Ip;
OdeCoeff{4}  = Uv;
OdeCoeff{5}  = Iv;
OdeCoeff{6}  = Is;
OdeCoeff{7}  = A2;
OdeCoeff{8}  = L;
OdeCoeff{9}  = C;
OdeCoeff{10} = [];
OdeCoeff{11} = Ug0;
Um           = 0.5*(Uv+Up-0.04);
Im           = TunnelDiode_Ir_dIr_d2Ir(Um,OdeCoeff);
R            = (Ug0 - Um) /Im;
OdeCoeff{10} = R;

F         = 1/(2*pi*sqrt(L*C));
NbrPer    = 1000;
T         = 1/F;

Ti        = 0;
Tf        = NbrPer * T;
NbrPasPer = 10;
tspan     = [0 Tf];

display('If Tol > 1e-12  NaN difficulties stop the calculation.')

tol       = 1e-12;  % Nan difficulties if larger value 
RelTolDef = tol;
AbsTolDef = tol;

InfoStatDef    = true;
InfoDynDef     = true;

NbrStgDef    = 3;
MinNbrStgDef = 3;
MaxNbrStgDef = 7;

options = [];
options = rdpset(options,'RelTol',RelTolDef);
options = rdpset(options,'AbsTol',AbsTolDef);
options = rdpset(options,'NbrStg',NbrStgDef);
options = rdpset(options,'MinNbrStg',MinNbrStgDef);
options = rdpset(options,'MaxNbrStg',MaxNbrStgDef);

% Voltage - current of the tunnel diode.
Umin = -0.01;
Umax = 0.48;
dU   = 0.001;
Ud   = [Umin:dU:Umax]';

[Id,Id1,Id2] = TunnelDiode_Ir_dIr_d2Ir(Ud,OdeCoeff);

Fig = 1;
figure(Fig); Fig = Fig+1;
subplot(3,1,1)
plot(Ud,Id);
hold on
grid on
plot(Up,Ip,'or')
plot(Uv,Iv,'or')
plot(Um,Im,'og')
title('Voltage - Current of the tunnel diode ');
xlabel('Voltage   [V]');
ylabel('Current  [ A ]');

subplot(3,1,2)
plot(Ud,Id1);
hold on
plot(Ud,Id1+1/R,'r')
grid on
title(' Dynamic conductance dI/dU in function of Voltage');
xlabel('Voltage   [V]');
ylabel('Conductance  [ 1/Ohm ]');

subplot(3,1,3)
plot(Ud,Id2)
title(' d2I/dU2 ');
xlabel('Voltage   [V]');
ylabel(' d2I/dU2  [ 1/(Ohm * V) ]');
grid on
% ----------------------------


tic
[tout,yout,Stats] = radau(FcnDef,tspan,y0,options,OdeCoeff);    
toc

InfoStat = Stats.Stat;
InfoDyn  = Stats.Dyn;

figure
plot(tout,real(yout(:,1)),'b')
grid on
hold on
plot(tout,real(yout(:,1)),'.b')
grid on   
title('OscDiodeTunnel, Ud')
   
figure 
plot(real(yout(:,1)),real(yout(:,2)),'b')
title('OscDiodeTunnel, y(2) versus y(1)')

if InfoDynDef
  figure
    
  subplot(4,1,1)
  plot(tout,real(yout(:,1)),'b')   
  hold on
  plot(tout,real(yout(:,1)),'.b')
  grid on    
  title('OscDiodeTunnel, u  in blue ')
   
  subplot(4,1,2)
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept);
  hold on
  grid on
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept,'.k');
  plot(InfoDyn.hreject_t,InfoDyn.hreject,'xr')
  title('Length of steps in function of time')
    
  subplot(4,1,3)
  plot(InfoDyn.Newt_t,InfoDyn.NewtNbr)
  grid on
  title('Newton iteration number in function of time')
    
  subplot(4,1,4)
  plot(InfoDyn.Newt_Step,InfoDyn.NewtNbr)
  hold on
  plot(InfoDyn.Jac_Step,0.5,'.b');
  title('Newton iteration number in function of steps and Jacobian')
    
  figure
  subplot(2,1,1)
  plot(InfoDyn.NbrStg_t,InfoDyn.NbrStg,'or')
  grid on
  title('Number of Stages versus Time')
  subplot(2,1,2)
  plot(InfoDyn.NbrStg_Step,InfoDyn.NbrStg,'ob')
  grid on
  Nbr1 = length(find(InfoDyn.NbrStg == 1));
  Nbr3 = length(find(InfoDyn.NbrStg == 3));
  Nbr5 = length(find(InfoDyn.NbrStg == 5));
  Nbr7 = length(find(InfoDyn.NbrStg == 7));
  NbrStg_1_3_5_7 = [Nbr1,Nbr3,Nbr5,Nbr7]  
end
if InfoStatDef
  InfoStat = InfoStat
end

options = [];
options = odeset(options,'RelTol',RelTolDef);
options = odeset(options,'AbsTol',AbsTolDef);

[tout,yout,Stat] = ode15s(FcnDef,tspan,y0,options,OdeCoeff);    
toc

figure
plot(tout,real(yout(:,1)),'b')
grid on
hold on
plot(tout,real(yout(:,1)),'.b')
grid on   
title('OscDiodeTunnel, Ud')
   
figure 
plot(real(yout(:,1)),real(yout(:,2)),'b')
title('OscDiodeTunnel, y(2) versus y(1)')
