% Driver for the Robertson's problem
% Differential equations:
% du/dt = -0.04*u + 1e4*v.*w;
% dv/dt =  0.04*u - 1e4*v.*w - 3e7*v.*v;
% dw/dt =           3e7*v.*v;
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%    
% See also http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
% ---------------------------

clc
close all
clear all

addpath D:\RDPSolver

ProblemName = 'Robertson';

% Function and jacobian
FcnDef = @RobertsonFcn;
JacDef = @RobertsonJac;

% Initial conditions
y0       = [1;0;0];

Ti       = 0;
Tf       = 1e11;
tspan    = [Ti Tf]; 

tol        = 1e-12;
RelTolDef  = tol;
AbsTolDef  = 1e-6 * RelTolDef;

InfoStatDef    = true;
InfoDynDef     = true;

NbrStgDef    = 3; 
MinNbrStgDef = 3;
MaxNbrStgDef = 7;

options = [];
options = rdpset(options,'RelTol',RelTolDef);
options = rdpset(options,'AbsTol',AbsTolDef);
options = rdpset(options,'NbrStg',NbrStgDef);
options = rdpset(options,'MinNbrStg',MinNbrStgDef);
options = rdpset(options,'MaxNbrStg',MaxNbrStgDef);
options = rdpset(options,'JacFcn',JacDef);

tic
[tout,yout,Stats] = radau(FcnDef,tspan,y0,options);
T_radau = toc

tspan =  [0 1e5];
tic
[t,y,Stats] = radau(FcnDef,tspan,y0,options);
T_radau = toc

InfoStat = Stats.Stat;
InfoDyn  = Stats.Dyn;

figure
subplot(2,1,1)
plot(tout,real(yout(:,1)),'b')  
hold on
plot(tout,real(yout(:,1)),'.b')
grid on
title('Complete Graphic')
subplot(2,1,2)
plot(t,real(y(:,1)),'b')  
hold on
plot(t,real(y(:,1)),'.b')
grid on
title('Transient Part')

if InfoDynDef
  figure    
  subplot(4,1,1)
  plot(tout,real(yout(:,1)))
  grid on
  hold on
  plot(tout,real(yout(:,1)),'.r')
  title(['Robertson component no 1'])
    
  subplot(4,1,2)
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept);
  hold on
  grid on
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept,'.k');
  plot(InfoDyn.hreject_t,InfoDyn.hreject,'xr')
  xlim([tspan(1) tspan(end)])
  title('Length of steps in function of time')
    
  subplot(4,1,3)
  plot(InfoDyn.Newt_t,InfoDyn.NewtNbr)
  grid on
  xlim([tspan(1) tspan(end)])
  title('Newton iteration number in function of time')
    
  subplot(4,1,4)
  plot(InfoDyn.Newt_Step,InfoDyn.NewtNbr)
  hold on
  plot(InfoDyn.Jac_Step,0.5,'.b');
  title('Newton iteration number in function of steps and Jacobian')
    
  figure
  subplot(2,1,1)
  plot(InfoDyn.NbrStg_t,InfoDyn.NbrStg,'or')
  grid on
  title('Number of Stages versus Time')
  subplot(2,1,2)
  plot(InfoDyn.NbrStg_Step,InfoDyn.NbrStg,'ob')
  grid on
  title('Number of Stages versus Steps Number')
  Nbr1 = length(find(InfoDyn.NbrStg == 1));
  Nbr3 = length(find(InfoDyn.NbrStg == 3));
  Nbr5 = length(find(InfoDyn.NbrStg == 5));
  Nbr7 = length(find(InfoDyn.NbrStg == 7));
  NbrStg_1_3_5_7 = [Nbr1,Nbr3,Nbr5,Nbr7]
end
if InfoStatDef
  InfoStat
end

options = [];
options = odeset(options,'RelTol',RelTolDef);
options = odeset(options,'AbsTol',AbsTolDef);
options = odeset(options,'Jacobian',JacDef);

tic
[tout,yout,Stat] = ode15s(FcnDef,tspan,y0,options);
T_ode15s = toc

tspan =  [0 1e5];
tic
[t,y,Stat] = ode15s(FcnDef,tspan,y0,options);
T_ode15s = toc


figure
subplot(2,1,1)
plot(tout,real(yout(:,1)),'b')  
hold on
plot(tout,real(yout(:,1)),'.b')
grid on
title('Complete Graphic')
subplot(2,1,2)
plot(t,real(y(:,1)),'b')  
hold on
plot(t,real(y(:,1)),'.b')
grid on
title('Transient Part')

