% Driver for the stiff example, Vol 2 page 2 
% Differential equations:
% dy/dt = -alpha*(y-cos(x))
% Initial condition : y(0) = 0
% ---------------------------
% See
%    E. Hairer S.P. Norsett G. Wanner
%    Solving Ordinary Differential Equations I
%    Nonstiff Problems
%    Springer Verlag
%    ISBN 3-540-17145-2, ISBN 0-387-17145-2
%    
%    E. Hairer G. Wanner
%    Solving Ordinary Differential Equations II
%    Stiff and Differential-Algebraic Problems
%    Springer Verlag
%    ISBN 3-540-53775-9 ISBN 0-387-53775-9
%    
% See also http://www.unige.ch/~hairer/software.html
% ---------------------------
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
%   Denis Bichsel
%   Rue des Deurres 58
%   2000 Neuchtel
%   Suisse
%   dbichsel@infomaniak.ch
%   End of 2015
% ---------------------------

clear all
close all
clc

warning on

addpath D:\RDPSolver

NomProbleme = 'Stiff page 2';

% Function and jacobian
FcnDef = @Stiff;
JacDef = @StiffJac;
% No Mass

% Parameter
alpha = 50;

% Initial condition
y0 = 0.0;  

Ti    = 0;
Tf    = 2;
dt    = (Tf-Ti)/4;
tspan = [Ti:dt:Tf]; 

OdeCoeff    = alpha;

tol         = 1e-6;
RelTolDef   = tol;
AbsTolDef   = tol;

InfoStatDef   = true;
InfoDynDef    = true;

NbrStgDef    = 3; 
MinNbrStgDef = 1;
MaxNbrStgDef = 7;

options = [];
options = rdpset(options,'RelTol',RelTolDef);
options = rdpset(options,'AbsTol',AbsTolDef);
options = rdpset(options,'NbrStg',NbrStgDef);
options = rdpset(options,'MinNbrStg',MinNbrStgDef);
options = rdpset(options,'MaxNbrStg',MaxNbrStgDef);
options = rdpset(options,'JacFcn',JacDef);

tic
[tout,yout,Stats] = radau(FcnDef,tspan,y0,options,OdeCoeff);    
T_radau = toc

InfoStat = Stats.Stat;
InfoDyn  = Stats.Dyn;

c        = alpha;
te       = [tspan(1):0.01:tspan(end)];
ExactSol = c^2/(1+c^2)*(- exp(-c*te) + cos(te)) + c/(1+c^2)*sin(te);

figure
plot(tout,real(yout(:,1)))
hold on
plot(tout,real(yout(:,1)),'.r')
grid on
hold on
plot(te,ExactSol,'k')
  
te        = tout;
ExactSol =  c^2/(1+c^2)*(- exp(-c*te) + cos(te)) + c/(1+c^2)*sin(te);
figure
plot(tout,real(yout(:,1))- ExactSol)
grid on
 
if InfoDynDef
  figure    
  subplot(4,1,1)
  plot(tout,real(yout(:,1)))
  grid on
  hold on
  plot(tout,real(yout(:,1)),'.r')
  title('Stiff page 2')
    
  subplot(4,1,2)
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept);
  hold on
  grid on
  semilogy(InfoDyn.haccept_t,InfoDyn.haccept,'.k');
  plot(InfoDyn.hreject_t,InfoDyn.hreject,'xr')
  xlim([tspan(1) tspan(end)])
  title('Length of steps in function of time')
    
  subplot(4,1,3)
  plot(InfoDyn.Newt_t,InfoDyn.NewtNbr)
  grid on
  xlim([tspan(1) tspan(end)])
  title('Newton iteration number in function of time')
    
  subplot(4,1,4)
  plot(InfoDyn.Newt_Step,InfoDyn.NewtNbr)
  hold on
  plot(InfoDyn.Jac_Step,0.5,'.b');
  title('Newton iteration number in function of steps and Jacobian')
    
  figure
  subplot(2,1,1)
  plot(InfoDyn.NbrStg_t,InfoDyn.NbrStg,'or')
  grid on
  title('Number of Stages versus Time')
  subplot(2,1,2)
  plot(InfoDyn.NbrStg_Step,InfoDyn.NbrStg,'ob')
  grid on
  title('Number of Stages versus Steps Number')
  Nbr1 = length(find(InfoDyn.NbrStg == 1));
  Nbr3 = length(find(InfoDyn.NbrStg == 3));
  Nbr5 = length(find(InfoDyn.NbrStg == 5));
  Nbr7 = length(find(InfoDyn.NbrStg == 7));
  NbrNbr_1_3_5_7 = [Nbr1,Nbr3,Nbr5,Nbr7]  
end
if InfoStatDef
  InfoStat = InfoStat
end
  
options = [];
options = odeset(options,'RelTol',RelTolDef);
options = odeset(options,'AbsTol',AbsTolDef);
options = odeset(options,'Jacobian',JacDef);

tic
[tout,yout,Stat] = ode15s(FcnDef,tspan,y0,options,OdeCoeff);    
T_ode15s = toc

c        = alpha;
te       = [tspan(1):0.01:tspan(end)];
ExactSol = c^2/(1+c^2)*(- exp(-c*te) + cos(te)) + c/(1+c^2)*sin(te);

figure
plot(tout,real(yout(:,1)))
hold on
plot(tout,real(yout(:,1)),'.r')
grid on
hold on
plot(te,ExactSol,'k')
  
te        = tout;
ExactSol =  c^2/(1+c^2)*(- exp(-c*te) + cos(te)) + c/(1+c^2)*sin(te);
figure
plot(tout,real(yout(:,1))- ExactSol)
grid on

