function [T, X] = full(t)
%FULL Convert tangent to a (full) tensor.
%
%   T = FULL(T) converts a tangent t to a (full) tensor. Use with care!
%   If nargout==2, we also return X as a by product.

% Geometric Hierchical Tucker 
% Copyright 2012 Bart Vandereycken, EPF Lausanne
% GPLv3 License, see COPYING.txt for details.

% Cell array to store the matrices U and dU (column basis for matricization) 
% for each node.
U = cell(1, t.x.nr_nodes);
dU = cell(1, t.x.nr_nodes);

x_is_leaf = t.x.is_leaf;

% Loop through all nodes, starting with the leaf nodes.
for ii=t.x.nr_nodes:-1:1
  
  if(x_is_leaf(ii))
    % Matrix U already known
    U{ii} = t.x.U{ii};
    dU{ii} = t.dx_h.U{ii};
    
  else
    % U12 = (U2 x U1) B12
    % dU12 = (dU2 x U1 + U2 x dU1) B12 + (U2 x U1) dB12
    % Find child nodes
    ii_left  = t.x.children(ii, 1);
    ii_right = t.x.children(ii, 2);

    U12 = ttm(t.x.B{ii}, {U{ii_left}, U{ii_right}}, [1 2]);
    dU12 = ttm(t.x.B{ii}, {dU{ii_left}, U{ii_right}}, [1 2]) + ...
      ttm(t.x.B{ii}, {U{ii_left}, dU{ii_right}}, [1 2]) + ...
      ttm(t.dx_h.B{ii}, {U{ii_left}, U{ii_right}}, [1 2]);
    
    U{ii} = matricize(U12, [1 2], 3, false);
    dU{ii} = matricize(dU12, [1 2], 3, false);
    
    
    % Clear variables to save memory
    %clear BUU;
    %U{ii_left} = [];
    %U{ii_right} = [];
  end 

end

% Vectorization of the full tensor is now contained in U{1}.
% Reshape to get full tensor
T = dematricize(dU{1}, size(t.x), t.x.dims{1}, [], false);
if nargout==2
  X = dematricize(U{1}, size(t.x), t.x.dims{1}, [], false);
end
