# ========================
# SPEECH ANALYSIS
# ========================
# Author: ngoet
# First version: 1.11.2017
# This version: 28.9.2019
# Replication for: Frech, Goet, Hug, 'Shirking and Slacking in Parliament'

# Custom functions
library(stargazer)
mod_stargazer <- function(output.file, ...) {
  output <- capture.output(stargazer(...))
  cat(paste(output, collapse = '\n'), '\n', file=output.file, append=F)
}

library(data.table)

spr_mean <- read.csv('data/regression_dfs/spr_mean.csv')
#plot
library(ggplot2)

participationRateFig <- ggplot() + geom_line(data=spr_mean,aes(x=Year,y=SpeechesN,lty=factor(Exit))) +
  theme_classic() +
  ylab('Debate participation') +
  theme(legend.position='bottom', legend.direction='horizontal',legend.title=element_blank()) +
  theme(axis.text.x = element_text(angle = 45, hjust = 1)) +
  scale_x_continuous(breaks=seq(min(spr_mean$Year),max(spr_mean$Year),1)) +
  # Election years
  geom_vline(xintercept=2003, linetype='dotted')  +
  geom_text(aes(x=2003, label='\nElection year', y=10), angle=90) +
  geom_vline(xintercept=2007, linetype='dotted')  +
  geom_text(aes(x=2007, label='\nElection year', y=10), angle=90) + 
  geom_vline(xintercept=2011, linetype='dotted')  +
  geom_text(aes(x=2011, label='\nElection year', y=10), angle=90)

pdf(file='figures/ParticipationByGroupBothHouses.pdf',width=6,height=4)
print(participationRateFig)
dev.off()
 
message(paste('Figure ParticipationByGroupBothHouses.pdf generated (timestamp: ',
  Sys.time(),')',sep=''))


# Regression analysis
# ====================
# Per parliament
message(paste('Running regression analysis for debate participation (timestamp: ',
  Sys.time(),')',sep=''))
library(data.table)

# Read in data
sp_leg47 <- read.csv('data/regression_dfs/sp_leg47.csv')
sp_leg48 <- read.csv('data/regression_dfs/sp_leg48.csv')
sp_leg4748 <- read.csv('data/regression_dfs/sp_leg4748.csv')
                                  
# Two indicator variables, tl48 and ret48. tl48 is equal to 1 if MPs are subject to a term-limit, and ret48 is equal to 1 if MPs decided to retire for other reasons. For all other MPs, even those failing to get reelected, the two variables are equal to 0 (btw, if we adopt this coding in a general way, then writing about 'continuing MPs' is a bit misleading: those with only 0s are simply running again.

# Slacking
sp_leg47 <- read.csv('data/regression_dfs/sp_leg47.csv')
sp_leg48 <- read.csv('data/regression_dfs/sp_leg48.csv')
sp_leg4748 <- read.csv('data/regression_dfs/sp_leg4748.csv')
                                  
# Two indicator variables, tl48 and ret48. tl48 is equal to 1 if MPs are subject to a term-limit, and ret48 is equal to 1 if MPs decided to retire for other reasons. For all other MPs, even those failing to get reelected, the two variables are equal to 0 (btw, if we adopt this coding in a general way, then writing about 'continuing MPs' is a bit misleading: those with only 0s are simply running again.

# Slacking
#########
library(MASS)

sp_leg47 <- unique(subset(sp_leg47,
  select=c(participationRate,PersonNumber,party_id,
    TermLimit,cohort,seniority47,seniority48,age47,retired,chamber)))
age <- sp_leg47$age47
seniority <- sp_leg47$seniority47


mod47 <- lm((participationRate) ~ TermLimit + age + seniority + 
  as.factor(party_id) + as.factor(chamber) - 1,data=sp_leg47)

sp_leg48 <- unique(subset(sp_leg48,
  select=c(participationRate,PersonNumber,party_id,TermLimit,
    cohort,seniority47,seniority48,age47,retired,chamber))
)

age <- sp_leg48$age47 + 4
seniority <- sp_leg48$seniority48
mod48 <- lm((participationRate) ~ TermLimit + age + seniority + as.factor(party_id) + as.factor(chamber) - 1,data=sp_leg48)

age <- sp_leg4748$age47 + 4
seniority <- sp_leg4748$seniority48
mod4748slacking <- lm((diff4748) ~ TermLimit + age + seniority + as.factor(party_id) + as.factor(chamber) - 1,data=sp_leg4748)

mod_stargazer('regTables/speechSlackingFedOLS.tex',mod47,mod48,mod4748slacking,
              title='OLS estimates of slacking in debates in the National Council and Council of States',
              label='table:speechSlackingFedOLS',
              model.names=T,
              single.row=T,
              column.labels = c('47th Leg.','48th Leg.','Diff. 48-47'),
              dep.var.labels = c('Debate participation rate','Diff. debate participation rate'),
              covariate.labels= c(
                               'Term-limited 48th Leg.',
                               'Age at 1st year of leg. session',
                               'Years served at 1st year of leg. session',
                               'CsP',
                               'CVP',
                               'EVP',
                               'FDP',
                               'FGr',
                               'Gru',
                               'SP',
                               'SVP',
                               'Upper House'
                               )
              )

# Save data
save(mod4748slacking, file='saved_models/mod4748slacking.RData')

message(paste('Regression analysis for debate participation completed (timestamp: ',
  Sys.time(),')',sep=''))

# Shirking
##########
speech_shirking_mean <- read.csv('data/regression_dfs/speech_shirking_mean.csv')

#plot
library(ggplot2)

shirkingPlot <- ggplot() + geom_line(data=speech_shirking_mean,aes(x=Year,y=MPposition,group=exit,linetype=factor(exit))) + 
  theme_classic() +
  ylab('Partyness') +
  theme(legend.position='bottom', legend.direction='horizontal',legend.title=element_blank()) +
  theme(axis.text.x = element_text(angle = 45, hjust = 1)) +
  scale_x_continuous(breaks=seq(1999,2011,1),labels=seq(1999,2011,1)) +
  # Election years
  geom_vline(xintercept=2003, linetype='dotted')  +
  geom_text(aes(x=2003, label='\nElection year', y=.45), angle=90) +
  geom_vline(xintercept=2007, linetype='dotted')  +
  geom_text(aes(x=2007, label='\nElection year', y=.45), angle=90) + 
  geom_vline(xintercept=2011, linetype='dotted')  +
  geom_text(aes(x=2011, label='\nElection year', y=.45), angle=90)

pdf(file='figures/shirking_ch_parl.pdf',width=6,height=4)
print(shirkingPlot)
dev.off()

message(paste('Figure shirking_ch_parl.pdf generated (timestamp: ',Sys.time(),')',sep=''))

# Regression analysis
####################
# Per parliament
message(paste('Running regression analyses for shirking in debates (timestamp: ',Sys.time(),')',sep=''))

ssh_leg47 <- read.csv('data/regression_dfs/ssh_leg47.csv')

age <- ssh_leg47$age47
seniority <- ssh_leg47$seniority47
colnames(ssh_leg47)[colnames(ssh_leg47)=='Term.limited'] <- 'Term limited'
mod47 <- lm(score ~`Term limited` + age + seniority + as.factor(party_id) + 
  as.factor(chamber) - 1,data=ssh_leg47)

ssh_leg48 <- read.csv('data/regression_dfs/ssh_leg48.csv')
seniority <- ssh_leg48$seniority48
age <- ssh_leg48$age47+4
colnames(ssh_leg48)[colnames(ssh_leg48)=='Term.limited'] <- 'Term limited'

mod48 <- lm(score ~ `Term limited` + age + seniority + as.factor(party_id) + 
  as.factor(chamber) - 1,data=ssh_leg48)

ssh_leg4748 <- read.csv('data/regression_dfs/ssh_leg4748.csv')
colnames(ssh_leg4748)[colnames(ssh_leg4748)=='Term.limited'] <- 'Term limited'
age <- ssh_leg4748$age47+4
seniority <- ssh_leg4748$seniority48
mod4748shirking <- lm(diff4748 ~ `Term limited` + 
  age + seniority + as.factor(party_id) + 
  as.factor(chamber)  - 1,data=ssh_leg4748)

scriptPath <- rstudioapi::getSourceEditorContext()$path
mod_stargazer('regTables/speechShirkingFedOLS.tex',mod47,mod48,mod4748shirking,
              title='OLS estimates of shirking in debates in the National Council and Council of States',
              label='table:speechShirkingFedOLS',
              model.names=T,
              single.row=T,
              column.labels = c('47th Leg.','48th Leg.','Diff. 48-47'),
              dep.var.labels = c('Partyness','Diff. partyness'),
              covariate.labels= c('Term-limited 48th Leg.',
                                  'Age at 1st year of leg. session',
                                  'Years served at 1st year of leg. session',
                                  'CsP',
                                  'CVP',
                                  'EVP',
                                  'FDP',
                                  'FGr',
                                  'Gru',
                                  'SP',
                                  'SVP',
                                  'Upper House')
)

# Save model
save(mod4748shirking,file='saved_models/mod4748shirking.RData')
message(paste('Running regression analyses for shirking in debates completed (timestamp: ',Sys.time(),')',sep=''))


# Robustness checks
###################
# Run the models in long format, i.e. with an indicator variable for the year
# in which the legislator retired because of a term limit
# Shirking
# Store data
df_shirking <- read.csv('data/regression_dfs/df_shirking.csv')
colnames(df_shirking)[colnames(df_shirking)=='Term.limited'] <- 'Term limited'
df_shirking$Time <- ifelse(df_shirking$term=='l48', 1, 0)
speech_shirking_mod <- lm(score ~ 
  Time*`Term limited` + 
  age + 
  seniority + 
  as.factor(party_id) + 
  as.factor(chamber) - 1,
  data=df_shirking)
  
# Slacking
df_slacking <- read.csv('data/regression_dfs/slacking_df.csv')
colnames(df_slacking)[colnames(df_slacking)=='Term.limited'] <- 'Term limited'
df_slacking$Time <- ifelse(df_slacking$term=='l48', 1, 0)
speech_slacking_mod <- lm((participationRate) ~ 
                            Time*`Term limited` + 
                            age + 
                            seniority + 
                            as.factor(party_id) + 
                            as.factor(chamber) - 1,
                          data=df_slacking)

scriptPath <- rstudioapi::getSourceEditorContext()$path
mod_stargazer('regTablesRobustness/speechShirkingSlackingFedOLS_extra_mod.tex',
              speech_shirking_mod,
              speech_slacking_mod,
              title='OLS estimates of shirking in debates in the National Council and Council of States (additional models)',
              label='table:speechShirkingFedOLS',
              model.names=T,
              single.row=T,
              column.labels = c('Shirking','Slacking'),
              dep.var.labels = c('Partyness','Debate participation rate'),
              covariate.labels= c('Time',
                                  'Term-limited 48th Leg.',
                                  'Age at 1st year of leg. session',
                                  'Years served at 1st year of leg. session',
                                  'CsP',
                                  'CVP',
                                  'EVP',
                                  'FDP',
                                  'FGr',
                                  'Gru',
                                  'LdT',
                                  'LPS',
                                  'PdA',
                                  'SP',
                                  'SVP',
                                  'Upper House',
                                  'Time x Term limited')
)

save(speech_shirking_mod,file='saved_models/speechShirkingFedOLS_extra_mod.Rdata')
save(speech_slacking_mod,file='saved_models/speechSlackingFedOLS_extra_mod.Rdata')
