# ========================
# VOTES ANALYSIS
# ========================
# Author: ngoet
# First version: 1.11.2017
# This version: 6.3.2020
# Replication for: Frech, Goet, Hug, 'Shirking and Slacking in Parliament'


#custom functions
library(stargazer)
mod_stargazer <- function(output.file, ...) {
  output <- capture.output(stargazer(...))
  cat(paste(output, collapse = '\n'), '\n', file=output.file, append=F)
}

vote_abstention_rates_mean <- read.csv('data/regression_dfs/vote_abstention_rates_mean.csv')

#plot abstention rates
library(ggplot2)
abstention_plot <- ggplot() + geom_line(data=vote_abstention_rates_mean,aes(x=Year,y=abstention_rate,group=Exit,linetype=factor(Exit))) + 
  theme_classic() +
  ylab('Share of abstentions') +
  theme(legend.position='bottom', legend.direction='horizontal',legend.title=element_blank()) +
  scale_x_continuous(breaks=vote_abstention_rates_mean$Year, labels=vote_abstention_rates_mean$Year) + 
  theme(axis.text.x = element_text(angle = 45, hjust = 1)) +
  ggtitle('Abstention') +
  # Election years
  geom_vline(xintercept=2003, linetype='dotted')  +
  geom_text(aes(x=2002, label='\nElection year', y=.045), angle=90) +
  geom_vline(xintercept=2007, linetype='dotted')  +
  geom_text(aes(x=2006, label='\nElection year', y=.045), angle=90) + 
  geom_vline(xintercept=2011, linetype='dotted')  +
  geom_text(aes(x=2010, label='\nElection year', y=.045), angle=90)

#plot no participation
vote_non_participation_rates_mean <- read.csv('data/regression_dfs/vote_non_participation_rates_mean.csv')

#plot abstention rates
library(ggplot2)
participationplot <- ggplot() + geom_line(data=vote_non_participation_rates_mean,aes(x=Year,y=NonParticipationRate,group=Exit,linetype=factor(Exit))) + 
  theme_classic() +
  ylab('Share of non-participation') +
  theme(legend.position='bottom', legend.direction='horizontal',legend.title=element_blank()) +
  scale_x_continuous(breaks=vote_non_participation_rates_mean$Year, labels=vote_non_participation_rates_mean$Year) + 
  theme(axis.text.x = element_text(angle = 45, hjust = 1)) +
  
  ggtitle('Non-participation (absence)') +
  # Election years
  geom_vline(xintercept=2003, linetype='dotted')  +
  geom_text(aes(x=2002, label='\nElection year', y=.25), angle=90) +
  geom_vline(xintercept=2007, linetype='dotted')  +
  geom_text(aes(x=2006, label='\nElection year', y=.25), angle=90) + 
  geom_vline(xintercept=2011, linetype='dotted')  +
  geom_text(aes(x=2010, label='\nElection year', y=.25), angle=90)

#combine plots
library(gridExtra)

pdf(file='figures/votes_slacking.pdf',width=8,height=4)
grid.arrange(abstention_plot, participationplot,ncol=2)
dev.off()

######################
#diff-in-diff analysis
######################
#1. Abstention rates
#separate by legislatures
#aggregate by legislature instead
votes_abstention_aggregated47 <- read.csv('data/regression_dfs/votes_abstention_aggregated47.csv')

mod_leg47 <- lm(AbstentionRate ~  TermLimit + age + seniority + as.factor(PartyNumber) - 1, data=votes_abstention_aggregated47)

summary(mod_leg47)

#model for 48th legislature
votes_abstention_aggregated48 <- read.csv('data/regression_dfs/votes_abstention_aggregated48.csv')

mod_leg48 <- lm(AbstentionRate ~ TermLimit + age + seniority + as.factor(PartyNumber) - 1, data=votes_abstention_aggregated48)

summary(mod_leg48)

votes_abstention_aggregated4748 <- read.csv('data/regression_dfs/votes_abstention_aggregated4748.csv')

diffmod47_48slackingAbstention <- lm(diff4748 ~ TermLimit + age + seniority + as.factor(PartyNumber) -1,data=votes_abstention_aggregated4748)

#save
save(diffmod47_48slackingAbstention, file = 'saved_models/diffmod47_48slackingAbstention.RData')

mod_stargazer('regTables/slackingVotesAbstentionsLH.tex',mod_leg47,mod_leg48,diffmod47_48slackingAbstention,
              title='OLS estimates of slacking in votes in the National Council (Abstentions)',
              label='table:slackingVotesAbstentionsLH',
              model.names=T,
              single.row=T,
              column.labels = c('47th Leg.','48th Leg.','Diff. 48-47'),
              dep.var.labels = c('Prop. of abstentions','Diff. prop. abstentions'),
              covariate.labels= c('Term-limited 48th Leg.',
                                  'Age at 1st year of leg. session',
                                  'Years served at 1st year of leg. session',
                                  'CsP',
                                  'CVP',
                                  'EVP',
                                  'FDP',
                                  'FGr',
                                  'Gru',
                                  'LPS',
                                  'PdA',
                                  'SP',
                                  'SVP')
)

##################
#non-participation
##################
votes_no_participation_aggregated47 <- read.csv('data/regression_dfs/votes_no_participation_aggregated47.csv')

mod_leg47 <- lm(NonParticipationRate ~ TermLimit + age + seniority + as.factor(PartyNumber) - 1, data=votes_no_participation_aggregated47)

summary(mod_leg47)

# model for 48th legislature
votes_no_participation_aggregated48 <- read.csv('data/regression_dfs/votes_no_participation_aggregated48.csv')

age <- votes_no_participation_aggregated48$age47+4
seniority <- votes_no_participation_aggregated48$seniority48

mod_leg48 <- lm(NonParticipationRate ~ TermLimit + age + seniority + as.factor(PartyNumber) - 1, data=votes_no_participation_aggregated48)

summary(mod_leg48)

# model for 47th legislature
votes_no_participation_aggregated4748 <- read.csv('data/regression_dfs/votes_no_participation_aggregated4748.csv')

age <- votes_no_participation_aggregated4748$age47+4
seniority <- votes_no_participation_aggregated4748$seniority48

diffmod47_48NoParticipation <- lm(diff4748 ~ TermLimit + age + seniority + as.factor(PartyNumber) -1,data=votes_no_participation_aggregated4748)
summary(diffmod47_48NoParticipation)

save(diffmod47_48NoParticipation,file='saved_models/diffmod47_48NoParticipation.RData')

mod_stargazer('regTables/slackingVotesNonParticipationLH.tex',mod_leg47,mod_leg48,diffmod47_48NoParticipation,
              title='OLS estimates of slacking in votes in the National Council (Non-Participation)',
              label='table:slackingVotesNonParticipationLH',
              model.names=T,
              single.row=T,
              column.labels = c('47th Leg.','48th Leg.','Diff. 48-47'),
              dep.var.labels = c('Prop. of abstentions','Diff. prop. abstentions'),
              covariate.labels= c('Term-limited 48th Leg.',
                                  'Age at 1st year of leg. session',
                                  'Years served at 1st year of leg. session',
                                  'CsP',
                                  'CVP',
                                  'EVP',
                                  'FDP',
                                  'FGr',
                                  'Gru',
                                  'LPS',
                                  'PdA',
                                  'SP',
                                  'SVP')
)

# Loyalty in votes
##################
loyaltyObj <- read.csv('data/regression_dfs/loyaltyObj.csv')

votes_no_participation_aggregated47 <- merge(votes_no_participation_aggregated47,loyaltyObj,by=c('PersonNumber','IdLegislativePeriod'))

mod_leg47 <- lm(loyalty ~ TermLimit + age + seniority + as.factor(PartyNumber) - 1, data=votes_no_participation_aggregated47)

summary(mod_leg47)

#model for 48th legislature
votes_no_participation_aggregated48 <- merge(votes_no_participation_aggregated48,loyaltyObj,by=c('PersonNumber','IdLegislativePeriod'))

mod_leg48 <- lm(loyalty ~ TermLimit + age + seniority + as.factor(PartyNumber) - 1, data=votes_no_participation_aggregated48)

summary(mod_leg48)

#model for diff-in-diff
loyaltyObjByLeg <- split(loyaltyObj,loyaltyObj$IdLegislativePeriod)

loyaltyObjByLeg <- merge(loyaltyObjByLeg[[1]],
                         loyaltyObjByLeg[[2]],
                         by='PersonNumber')

loyaltyObjByLeg$diff4748 <- loyaltyObjByLeg$loyalty.y - loyaltyObjByLeg$loyalty.x

loyaltyObjByLeg <- subset(loyaltyObjByLeg,select=
                            c(PersonNumber,
                              diff4748)
                          )
votes_no_participation_aggregated4748 <- merge(
  subset(votes_no_participation_aggregated4748,select=-diff4748),
  loyaltyObjByLeg,
  by = c('PersonNumber')
)

diffmod47_48VoteLoyalty <- lm(diff4748 ~ TermLimit + age + seniority + as.factor(PartyNumber) -1,data=votes_no_participation_aggregated4748)
summary(diffmod47_48VoteLoyalty)

save(diffmod47_48VoteLoyalty,file='saved_models/diffmod47_48VoteLoyalty.RData')

mod_stargazer('regTables/shirkingVotesLoyaltyLH.tex',mod_leg47,mod_leg48,diffmod47_48NoParticipation,
              title='OLS estimates of loyalty in votes in the National Council (Non-Participation)',
              label='table:shirkingVotesLoyaltyLH',
              model.names=T,
              single.row=T,
              column.labels = c('47th Leg.','48th Leg.','Diff. 48-47'),
              dep.var.labels = c('Loyalty','Diff. loyalty abstentions'),
              covariate.labels= c('Term-limited 48th Leg.',
                                  'Age at 1st year of leg. session',
                                  'Years served at 1st year of leg. session',
                                  'CsP',
                                  'CVP',
                                  'EVP',
                                  'FDP',
                                  'FGr',
                                  'Gru',
                                  'LPS',
                                  'PdA',
                                  'SP',
                                  'SVP')
)


#combine table
mod_stargazer('regTables/combinedShirkingSlackingVotesLH.tex',diffmod47_48slackingAbstention,diffmod47_48NoParticipation,diffmod47_48VoteLoyalty,
              title='OLS estimates of DiD of slacking and shirking in votes in the National Council',
              label='table:combinedShirkingSlackingVotesLH.tex',
              model.names=T,
              single.row=T,
              column.labels = c('Abstention','Non-participation','Loyalty'),
              dep.var.labels = c('Diff-in-diff (48th Leg. - 47th Leg.)'),
              covariate.labels= c('Term-limited',
                                  'Age',
                                  'Seniority',
                                  'CsP',
                                  'CVP',
                                  'EVP',
                                  'FDP',
                                  'FGr',
                                  'Gru',
                                  'LPS',
                                  'PdA',
                                  'SP',
                                  'SVP')
)

# ===============================
# Additional models
# ===============================
# Models of the form $Y_{it} = \alpha_i + \delta_t + \beta termlimited_{it} + \epsilon_{it}$

# 1. Abstention
# ================
library(dplyr)
df47 <- subset(votes_abstention_aggregated47, select = -c(seniority48, seniority, age))
df47 <- df47 %>% dplyr::rename(
  seniority = seniority47,
  age = age47
)

df47 <- aggregate(TermLimit ~ 
                    AbstentionRate + 
                    age + 
                    seniority + 
                    PartyNumber +
                    IdLegislativePeriod,
                  data = df47, 
                  FUN=max)

df48 <- subset(votes_abstention_aggregated48, 
               select = -c(seniority47, age47, seniority))

df48 <- df48 %>% dplyr::rename(
  seniority = seniority48
) 

df48 <- aggregate(TermLimit ~ 
                    AbstentionRate + 
                    age + 
                    seniority + 
                    PartyNumber +
                    IdLegislativePeriod,
                  data = df48, 
                  FUN=max)

df_abstention_combined <- rbind(df47, df48)
df_abstention_combined$Time <- ifelse(df_abstention_combined$IdLegislativePeriod==48, 1, 0)
abstention_mod <- lm(AbstentionRate ~  Time*TermLimit + age + seniority + as.factor(PartyNumber) - 1, data=df_abstention_combined)

# 2. Participation
# ================

df47 <- subset(votes_no_participation_aggregated47, select = -c(seniority48, seniority, age))
df47 <- df47 %>% dplyr::rename(
  age = age47,
  seniority = seniority47
)

df48 <- subset(votes_no_participation_aggregated48, 
               select = -c(seniority47, age47))

df48 <- df48 %>% dplyr::rename(
  age = age48,
  seniority = seniority48
) 

df_participation_combined <- rbind(df47, df48)
df_participation_combined$Time <- ifelse(df_participation_combined$IdLegislativePeriod==48, 1, 0)

participation_mod <- lm(NonParticipationRate ~ Time*TermLimit + age + seniority + as.factor(PartyNumber) - 1,data=df_participation_combined)

# Loyalty
# =================
loyalty_mod <- lm(loyalty ~ Time*TermLimit + age + seniority + as.factor(PartyNumber) - 1, data=df_participation_combined)

# Write out results
mod_stargazer('regTablesRobustness/combinedShirkingSlackingVotesLH_extra_mod.tex',abstention_mod,participation_mod,loyalty_mod,
              title='OLS estimates of slacking and shirking in votes in the National Council (additional model)',
              label='table:combinedShirkingSlackingVotesLH.tex',
              model.names=T,
              single.row=T,
              column.labels = c('Abstention','Non-participation','Loyalty'),
              covariate.labels= c('Time',
                                  'Term-limited',
                                  'Age',
                                  'Seniority',
                                  'CsP',
                                  'CVP',
                                  'EVP',
                                  'FDP',
                                  'FGr',
                                  'Gru',
                                  'LPS',
                                  'PdA',
                                  'SP',
                                  'SVP',
                                  'Time*Term-limited')
)

if(nrow(df_abstention_combined) != 2*nrow(votes_abstention_aggregated4748)){
  message('Test failed: Check that number of legislators in df_abstention_combined == number of legislators of votes_abstention_aggregated4748')
}

save(abstention_mod,file='saved_models/AbstentionVotesLH_extra_mod.Rdata')
save(participation_mod,file='saved_models/ParticipationVotesLH_extra_mod.Rdata')
save(loyalty_mod,file='saved_models/LoyaltyVotesLH_extra_mod.Rdata')


