#=------------------------------------------------------------------------------
    Auxiliary functions of Script_weak_cv.jl

    Details can be found in the article:
    [ ] A. Laurent, G. Vilmart
        Multirevolution integrators for differential equations
        with fast stochastic oscillations
        SIAM J. Sci. Comput. 42 (2020), no. 1, A115–A139.

    PLEASE CITE THE ABOVE PAPER WHEN USING THIS PROGRAM ! :-)
    Version of 9th October 2019

    Uses Julia 1.2.0
=#
#=------------------------------------------------------------------------------
    Auxiliary functions
=#
function complex_to_real_vector(v::Array{Complex{Float64},1})
    d=length(v)
    x=zeros(2*d)
    for i in 1:d
        x[2*i-1]=real(v[i])
        x[2*i]=imag(v[i])
    end
    return x
end


function complex_to_real_matrix(Mat::Array{Complex{Float64},2})
    s=size(Mat)[1]
    X=zeros(2*s,2*s)
    for i in 1:s
        for j in 1:s
            X[2*i-1,2*j-1]=real(Mat[i,j])
            X[2*i-1,2*j]=-imag(Mat[i,j])
            X[2*i,2*j]=real(Mat[i,j])
            X[2*i,2*j-1]=imag(Mat[i,j])
        end
    end
    return X
end


function divisors(N::Int64)
    l=[]
    for n=1:N
        if rem(N,n)==0
            l=vcat(l,n)
        end
    end
    return l
end


function Fourier_discrete(y::Array{Complex{Float64},1})
#=
    Author: Adrien Laurent
    Computes the discrete Fourier transform coefficients of exp(-A*t)*F(exp(A*t)*y).
    Input:
        y initial parameter
    Output:
        C_Fourier discrete Fourier transform coefficients (size K_FFT_x*K_FFT)
=#
    h=1.0/K_FFT
    x=0:h:1-h
    C_Fourier=zeros(dim,K_FFT)*im
    aux=zeros(dim,K_FFT)*im
    V=zeros(dim,K_FFT)*im
    for l in 1:K_FFT
        V[:,l]=diagm(exp.(-diag(A)*x[l]))*F(exp.(diag(A)*x[l]).*y)
    end
    for c in 1:dim
        aux[c,:]=fft(V[c,:])/K_FFT
    end
    for k in 1:K_FFT
        if k<=N_FFT
            C_Fourier[:,k]=aux[:,k+N_FFT+1]
        else
            C_Fourier[:,k]=aux[:,k-N_FFT]
        end
    end
    return C_Fourier
end


function Fourier_discrete_1(y::Array{Complex{Float64},1})
#=
    Author: Adrien Laurent
    Computes the discrete Fourier transform coefficients of exp(-A*t)*DF(exp(A*t)*y)(exp(A*t)*.).
    Input:
        y initial parameter
    Output:
        C_Fourier discrete Fourier transform coefficients (size (2*K_FFT_x)*(2*K_FFT_x)*K_FFT)
=#
    h=1.0/K_FFT
    x=0:h:1-h
    C_Fourier=zeros(2*dim,2*dim,K_FFT)*im
    aux=zeros(2*dim,2*dim,K_FFT)*im
    V=zeros(2*dim,2*dim,K_FFT)
    for l in 1:K_FFT
        V[:,:,l]=complex_to_real_matrix(diagm(exp.(-diag(A)*x[l])))*DF(exp.(diag(A)*x[l]).*y)*complex_to_real_matrix(diagm(exp.(diag(A)*x[l])))
    end
    for a in 1:2*dim
        for b in 1:2*dim
            aux[a,b,:]=fft(V[a,b,:])/K_FFT
        end
    end
    for p in 1:K_FFT
        if p<=N_FFT
            C_Fourier[:,:,p]=aux[:,:,p+N_FFT+1]
        else
            C_Fourier[:,:,p]=aux[:,:,p-N_FFT]
        end
    end
    return C_Fourier
end


#=------------------------------------------------------------------------------
    Numerical integrators
=#
function  Integrator_Fourier_weak_order_1(U0::Array{Complex{Float64},1},epsilon::Float64,N::Int64,Mmax::Int64)
#=
    Author: Adrien Laurent
    Approximates u(epsilon*S_S_NMax) with initial condition U0 and weak order 1 accuracy.
    Input:
        U0 initial condition
        epsilon parameter of equation
        N number of compositions
        Mmax number of iterations
    Output:
        yM approximation vector of NLS at time epsilon*S_NMax
=#
    yM=zeros(dim,Mmax+1)*im
    yM[:,1]=U0
    for M in 1:Mmax
        C_Fourier_0=Fourier_discrete(yM[:,M])
        yM[:,M+1]=yM[:,M]+epsilon*N*C_Fourier_0[:,N_FFT+1]
    end
    return yM
end


function  Integrator_Fourier_weak_order_2(U0::Array{Complex{Float64},1},epsilon::Float64,N::Int64,Mmax::Int64)
#=
    Author: Adrien Laurent
    Approximates u(epsilon*S_NMax) with initial condition U0 and weak order 2 accuracy.
    Input:
        U0 initial condition
        epsilon parameter of equation
        N number of compositions
        Mmax number of iterations
    Output:
        yM approximation vector of NLS at time epsilon*S_NMax
=#
    yM=zeros(dim,Mmax+1)*im
    yM[:,1]=U0
    Cov_alpha=1/sqrt(N)*Cov_alpha_aux
    alpha=zeros(K_FFT)*im
    for M in 1:Mmax
        C_Fourier_0=Fourier_discrete(yM[:,M])
        C_Fourier_0_real=zeros(2*dim,K_FFT)
        for k in 1:K_FFT
            C_Fourier_0_real[:,k]=complex_to_real_vector(C_Fourier_0[:,k])
        end
        alpha_aux=Cov_alpha*(2*bitrand(2*K_FFT) .-1)
        for k in 1:K_FFT
            alpha[k]=alpha_aux[2*k-1]+im*alpha_aux[2*k]
        end
        alpha[N_FFT+1]+=1
        C_Fourier_1_aux=Fourier_discrete_1(yM[:,M])
        C_Fourier_1=zeros(dim,2*dim,K_FFT)*im
        for  j in 1:dim
            C_Fourier_1[j,:,:]=C_Fourier_1_aux[2*j-1,:,:]+im*C_Fourier_1_aux[2*j,:,:]
        end
        aux0=C_Fourier_0*alpha
        aux1=zeros(dim)*im
        aux1+=C_Fourier_1[:,:,N_FFT+1]*C_Fourier_0_real[:,N_FFT+1]*(1/2+1/(3*N))
        for k in -N_FFT:N_FFT
            if k!=0
                K=k+N_FFT+1
                K_op=-k+N_FFT+1
                aux1+=C_Fourier_1[:,:,N_FFT+1]*C_Fourier_0_real[:,K]/(2*pi^2*k^2*N)-C_Fourier_1[:,:,K]*C_Fourier_0_real[:,N_FFT+1]/(2*pi^2*k^2*N)+C_Fourier_1[:,:,K]*C_Fourier_0_real[:,K_op]/(2*pi^2*k^2*N)
            end
        end
        yM[:,M+1]=yM[:,M]+epsilon*N*aux0+epsilon^2*N^2*aux1
    end
    return yM
end


function  Integrator_Fourier_weak_order_1_geometric(U0::Array{Complex{Float64},1},epsilon::Float64,N::Int64,Mmax::Int64,tol::Float64,K_fixed_point::Int64)
#=
    Author: Adrien Laurent
    Approximates u(epsilon*S_NMax) with initial condition U0 and weak order 1 accuracy. Preserves quadratic invariants.
    Input:
        U0 initial condition
        epsilon parameter of equation
        N number of compositions
        Mmax number of iterations
        tol tolerance of the fixed point iteration
        K_fixed_point max iterations of the fixed point iteration
    Output:
        yM approximation vector of NLS at time epsilon*S_NMax
=#
    yM=zeros(dim,Mmax+1)*im
    yM[:,1]=U0
    for M in 1:Mmax
        yaux=yM[:,M]
        counter=0
        while (counter<K_fixed_point&&(abs(norm(yaux)-norm(yM[:,M]))>tol||counter==0))
            C_Fourier_0=Fourier_discrete((yM[:,M]+yaux)/2)
            yaux=yM[:,M]+epsilon*N*C_Fourier_0[:,N_FFT+1]
            counter+=1
        end
        yM[:,M+1]=yaux
    end
    return yM
end


function  Integrator_Fourier_weak_order_2_geometric(U0::Array{Complex{Float64},1},epsilon::Float64,N::Int64,Mmax::Int64,tol::Float64,K_fixed_point::Int64)
#=
    Author: Adrien Laurent
    Approximates u(epsilon*S_NMax) with initial condition U0 and weak order 2 accuracy. Preserves quadratic invariants.
    Input:
        U0 initial condition
        epsilon parameter of equation
        N number of compositions
        Mmax number of iterations
        tol tolerance of the fixed point iteration
        K_fixed_point max iterations of the fixed point iteration
    Output:
        yM approximation vector of NLS at time epsilon*S_NMax
=#
    yM=zeros(dim,Mmax+1)*im
    yM[:,1]=U0
    Cov_alpha=1/sqrt(N)*Cov_alpha_aux
    alpha=zeros(K_FFT)*im
    for M in 1:Mmax
        yaux=yM[:,M]
        counter=0
        alpha_aux=Cov_alpha*(2*bitrand(2*K_FFT) .-1)
        for k in 1:N_FFT
            alpha[k]=alpha_aux[2*k-1]+im*alpha_aux[2*k]
            alpha[K_FFT+1-k]=alpha_aux[2*k-1]-im*alpha_aux[2*k]
        end
        alpha[N_FFT+1]=1+alpha_aux[K_FFT]+im*alpha_aux[K_FFT+1]
        while (counter<K_fixed_point&&(abs(norm(yaux)-norm(yM[:,M]))>tol||counter==0))
            C_Fourier_0=Fourier_discrete((yM[:,M]+yaux)/2)
            C_Fourier_0_real=zeros(2*dim,K_FFT)
            for k in 1:K_FFT
                C_Fourier_0_real[:,k]=complex_to_real_vector(C_Fourier_0[:,k])
            end
            C_Fourier_1_aux=Fourier_discrete_1((yM[:,M]+yaux)/2)
            C_Fourier_1=zeros(dim,2*dim,K_FFT)*im
            for  j in 1:dim
                C_Fourier_1[j,:,:]=C_Fourier_1_aux[2*j-1,:,:]+im*C_Fourier_1_aux[2*j,:,:]
            end
            aux0=C_Fourier_0*alpha
            aux1=zeros(dim)*im
            for k in -N_FFT:N_FFT
                if k!=0
                    K=k+N_FFT+1
                    aux1+=C_Fourier_1[:,:,N_FFT+1]*C_Fourier_0_real[:,K]/(2*pi^2*k^2*N)-C_Fourier_1[:,:,K]*C_Fourier_0_real[:,N_FFT+1]/(2*pi^2*k^2*N)
                end
            end
            yaux=yM[:,M]+epsilon*N*aux0+epsilon^2*N^2*aux1
            counter+=1
        end
        yM[:,M+1]=yaux
    end
    return yM
end
