#=------------------------------------------------------------------------------
    Julia script for showing the weak order of convergence of method A and B

    Details can be found in the article (see Section 5.1):
    [ ] A. Laurent, G. Vilmart
        Multirevolution integrators for differential equations
        with fast stochastic oscillations
        SIAM J. Sci. Comput. 42 (2020), no. 1, A115–A139.

    PLEASE CITE THE ABOVE PAPER WHEN USING THIS PROGRAM ! :-)
    Version of 9th October 2019

    Uses Julia 1.2.0
=#
#=------------------------------------------------------------------------------
     Parameters of the equation
=#
# Dimension of the SDE
dim=1

# Nonlinearity
function F(u)
    return im*u
end
function DF(u)
    return [0.0 -1.0 ; 1.0 0.0]
end
#= function F(u)
    return im*u+im*(real.(u).^3+imag.(u).^5).*u
end
function DF(u)
    return [-3*real(u[1])^2*imag(u[1])  -1-real(u[1])^3-6*imag(u[1])^5 ; 1+4*real(u[1])^3+imag(u[1])^5  5*imag(u[1])^4*real(u[1])]
end =#

# Oscillation matrix
A=zeros(1,1)*im
A[1,1]=2*im*pi

# Initial data
u_0=zeros(dim)*im
u_0[1]=1.0+0.0im

# Test function
phi(x)=2*real(x[1])+4*imag(x[1])
#=------------------------------------------------------------------------------
    Numerical parameters of the problem
=#
N_FFT=8     # number of Fourier coefficients
K_FFT=2*N_FFT+1
epsilon=1.0e-3
tol=1.0e-13     # fixed point tolerance for geometric method
K_fixed_point=100   # fixed point max iterations for geometric method
N_traj=10000    # number of trajectories per processor
Nmax=2^8    # value of N*M
#=------------------------------------------------------------------------------
     Functions
=#
using Random
using DelimitedFiles
using LinearAlgebra
#Pkg.add("FFTW")
using FFTW

include("Functions.jl")
#=------------------------------------------------------------------------------
     Initialize random number generator
=#
if length(ARGS)==0
    Random.seed!(0)
else
    Random.seed!(parse(Int64,ARGS[1]))
end
#=------------------------------------------------------------------------------
    Random variables constants
=#
L_alpha=zeros(2*K_FFT,2*K_FFT)
for p in -N_FFT:N_FFT
    P=p+N_FFT+1
    P_op=-p+N_FFT+1
    if p==0
        L_alpha[2*P-1:2*P,2*P-1:2*P]=[2/3 0.0; 0.0 0.0]
    else
        L_alpha[2*P-1:2*P,2*P-1:2*P]=[1/(2*pi^2*p^2) 0.0;0.0 1/(2*pi^2*p^2)]
        L_alpha[2*P-1:2*P,2*P_op-1:2*P_op]=[1/(2*pi^2*p^2) 0.0;0.0 -1/(2*pi^2*p^2)]
    end
end
Cov_alpha_aux=sqrt(L_alpha)
#=------------------------------------------------------------------------------
     Main script
=#
LN=divisors(Nmax)
lN=length(LN)
Ephiu=zeros(4,lN)

start = time()
for i=1:lN
    N=LN[i]
    println("N=",N)
    M=div(Nmax,N)
    Ephiunum1=0.0
    for j=1:N_traj
        U_composition1=Integrator_Fourier_weak_order_1(u_0,epsilon,N,M)[:,M+1]
        Ephiunum1+=phi(U_composition1)
    end
    Ephiu[1,i]=1/N_traj*Ephiunum1

    Ephiunum2=0.0
    for j=1:N_traj
        U_composition2=Integrator_Fourier_weak_order_2(u_0,epsilon,N,M)[:,M+1]
        Ephiunum2+=phi(U_composition2)
    end
    Ephiu[2,i]=1/N_traj*Ephiunum2

    Ephiunum1G=0.0
    for j=1:N_traj
        U_composition1G=Integrator_Fourier_weak_order_1_geometric(u_0,epsilon,N,M,tol,K_fixed_point)[:,M+1]
        Ephiunum1G+=phi(U_composition1G)
    end
    Ephiu[3,i]=1/N_traj*Ephiunum1G

    Ephiunum2G=0.0
    for j=1:N_traj
        U_composition2G=Integrator_Fourier_weak_order_2_geometric(u_0,epsilon,N,M,tol,K_fixed_point)[:,M+1]
        Ephiunum2G+=phi(U_composition2G)
    end
    Ephiu[4,i]=1/N_traj*Ephiunum2G
end
println(time() - start)
#=------------------------------------------------------------------------------
     Storing results
=#
if length(ARGS)==0
    writedlm("./Data/data_num.txt", Ephiu)
else
    writedlm(string("./results/res_",ARGS[1],".txt"), Ephiu)
end

println("end")
