%FE_HMM2D   two-dimensional heterogeneous multiscale finite element method method for elliptic problems.
%   FE_HMM2D solves the equation
%     - div(a*grad(u)) = f in Omega
%                    u = u_D on the Dirichlet boundary
%               d/dn u = g   on the Neumann boundary
%   with a highly oscillatory tensor a(x)
%   on a geometry described by triangles and parallelograms, respectively
%   and presents the solution graphically.
%
%
%   The code is available at http://iacs.epfl.ch/anmc/ and described in 
%   further detail in 
%
%   A. Abdulle and A. Nonnenmacher
%   "A short and versatile finite element multiscale code for
%   homogenization problems"
%   Computer Methods in Applied Mechanics and Engineering,
%   Vol. XXXXXXX, p. XXXXXXX
%
%   Please cite this article in any publication describing research
%   performed using the software.
%
%
%
%   The structure of the directories is as follows:
%   fe-hmm          
%   +---examples
%   | +---elliptic    contains main file and files for the problem specific setup
%   | | +---data      contains geometry data files
%   | +---parabolic   contains main file and files for the problem specific setup  
%   | | +---data      contains geometry data files
%   | +---tools       contains problem independent algorithm files
%   
%   To adapt the program to a given setup of the equation, the user has to 
%   modify the geometry data files in ./data and the M-files
%   <f.m>, <g.m>, <u_d.m> and <tensor_a.m>. They have to be in the same
%   directory as <fe_hmm2d.m>.
%
%
%   This code is based in part on
%   [1] J. Alberty, C. Carstensen & S. Funken
%       "Remarks around 50 lines of Matlab: short finite element implementation",
%        Numerical Algorithms, Springer, 1999, 20, 117-137
%
%   Email           : assyr.abdulle@epfl.ch and achim.nonnenmacher@epfl.ch
%   Last updated    : 04/29/2009 with MATLAB 7.4
%
%   FE_HMM2D is Copyright (C) 2009 A. Abdulle and A. Nonnenmacher. 
%   The software is provided free for non-commercial use unter the terms of 
%   the GNU General Public License. See "copyright.m" for full details.   
%

%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Parallel version: type Matlabpool 
if matlabpool('size')==0
    disp('Warning: First use the command matlabpool to start the parallel computing pool.')
end

% check if tools directory was added to the path
if(exist('hmm_microstima_nonlin.m')~=2)
   error('HMMFEM:path', strcat('Please add "tools" directory to the Matlab path.\n',...
       'You can do that with the command:\n',...
       'addpath(''../../tools'');'));
end

% check if FE-HMM has been installed
if(exist('shapefunction_quad_lin.m')~=2)
   error('HMMFEM:path', strcat('To use FE_HMM2D_NONLIN, please install first FE_HMM2D'));
end


% Variables
epsilon =5e-2;

% Discretization points per space dimension of the micro domain
NMicro  =4;

% Size of the micro sampling domain
delta= epsilon;


% Initialisation
Coordinates=load('data/coordinates.dat');
Elements3  =load('data/elements3.dat');
Elements4  =load('data/elements4.dat');
Neumann    =load('data/neumann.dat');
Dirichlet  =load('data/dirichlet.dat');

%tolerance for Newton iterations
tol = 1e-8;

bctype='periodic';
% bctype='dirichlet';

tic

% initial guess for the Newton iterations
u = sparse(size(Coordinates,1),1);
incr=[];

incr0=1;
% Loop for Newton iterations
k=0;
while incr0>tol
k=k+1;
uold=u;

FreeNodes=setdiff(1:size(Coordinates,1),unique(Dirichlet));
b = sparse(size(Coordinates,1),1);

% HMM initialization, microgrid with quadrilaterals, periodic constraints
MicroElements=micromesh_elements(NMicro);
Constraints=make_constraints(NMicro, bctype);
NMicroB=NMicro;
% Define coarser micro meshes for the first 3 Newton iterations.
if (k<=3)
    NMicroB=max(2,NMicro/4);
end
MicroElementsB=micromesh_elements(NMicroB);
ConstraintsB=make_constraints(NMicroB, bctype);

% Assembly
% For meshes with many DOF this can be very slow due to Matlab's way of 
% handling sparse matrices. An optimized code can be found at the very
% bottom of this file.

% matrix B is for Newton iterations

if matlabpool('size')==0
%disp('NON PARALLEL COMPUTING');
A = sparse(size(Coordinates,1),size(Coordinates,1));
B = sparse(size(Coordinates,1),size(Coordinates,1));
for j=1:size(Elements3,1)
    [MA,MB]=hmm_stima_tri_nonlin_jac(u(Elements3(j,:)),NMicro,NMicroB,epsilon,...
        MicroElements,MicroElementsB,Coordinates(Elements3(j,:),:),Constraints,ConstraintsB,delta, bctype);
    A(Elements3(j,:), Elements3(j,:)) = A(Elements3(j,:), Elements3(j,:))+MA;
    B(Elements3(j,:), Elements3(j,:)) = B(Elements3(j,:), Elements3(j,:))+MB;
end

for j=1:size(Elements4,1)
    [MA,MB]=hmm_stima_quad_nonlin_jac(u(Elements4(j,:)),NMicro,NMicroB,epsilon,...
        MicroElements,MicroElementsB,Coordinates(Elements4(j,:),:),Constraints,ConstraintsB,delta, bctype);
    A(Elements4(j,:), Elements4(j,:)) = A(Elements4(j,:), Elements4(j,:))+MA;
    B(Elements4(j,:), Elements4(j,:)) = B(Elements4(j,:), Elements4(j,:))+MB;
end
else
%disp('PARALLEL COMPUTING');
valuepar=[];
parfor j=1:size(Elements3,1)
    [MA,MB]=hmm_stima_tri_nonlin_jac(u(Elements3(j,:)),NMicro,NMicroB,epsilon,...
         MicroElements,MicroElementsB,Coordinates(Elements3(j,:),:),Constraints,ConstraintsB,delta, bctype);
    tmpi=repmat(Elements3(j,:),1,3);
    tmpj=repmat(Elements3(j,:),3,1);
    newpar=[tmpi(:),tmpj(:),MA(:),MB(:)]';
    valuepar=[valuepar,newpar];
end
parfor j=1:size(Elements4,1)
    [MA,MB]=hmm_stima_quad_nonlin_jac(u(Elements4(j,:)),NMicro,NMicroB,epsilon,...
         MicroElements,MicroElementsB,Coordinates(Elements4(j,:),:),Constraints,ConstraintsB,delta, bctype);
    tmpi=repmat(Elements4(j,:),1,4);
    tmpj=repmat(Elements4(j,:),4,1);
    newpar=[tmpi(:),tmpj(:),MA(:),MB(:)]';
    valuepar=[valuepar,newpar];
end
A=sparse(valuepar(1,:),valuepar(2,:),valuepar(3,:),size(Coordinates,1),size(Coordinates,1)); 
B=sparse(valuepar(1,:),valuepar(2,:),valuepar(4,:),size(Coordinates,1),size(Coordinates,1)); 
end


% Volume Forces
for j=1:size(Elements3,1)
  b(Elements3(j,:)) = b(Elements3(j,:))+ ...
      det([1,1,1; Coordinates(Elements3(j,:),:)'])* ...
      f(sum(Coordinates(Elements3(j,:),:))/3)/6;
end

for j=1:size(Elements4,1)
  b(Elements4(j,:)) = b(Elements4(j,:))+ ...
      det([1,1,1; Coordinates(Elements4(j,1:3),:)'])* ...
      f(sum(Coordinates(Elements4(j,:),:))/4)/4;
end

% Neumann conditions
for j = 1 : size(Neumann,1)
  b(Neumann(j,:))=b(Neumann(j,:)) + norm(Coordinates(Neumann(j,1),:)- ...
      Coordinates(Neumann(j,2),:)) * g(sum(Coordinates(Neumann(j,:),:))/2)/2;
end

% Dirichlet conditions 
u = zeros(size(Coordinates,1),1);
u(unique(Dirichlet)) = u_d(Coordinates(unique(Dirichlet),:));
b = b - A * u;

% NEWTON ITERATIONS
u(FreeNodes) = uold(FreeNodes)- ...
       (A(FreeNodes,FreeNodes)+B(FreeNodes,FreeNodes))\...
       (A(FreeNodes,FreeNodes)*uold(FreeNodes)-b(FreeNodes));

   
incr0=max(abs(u-uold));
incr=[incr,incr0];
end

toc

disp('List of Newton iteration increments:')
disp(sprintf('%g    ',incr));


% graphic representation
show(Elements3,Elements4,Coordinates,full(u));
title('nonlinear elliptic problem');

enorm=sqrt(u'*A*u);
disp(sprintf('energy norm is |u|=%g',enorm));
disp(sprintf('max         is |u|=%g',max(u)));
