%FE_HMM2D_PARA   two-dimensional heterogeneous multiscale finite element method method for the heat equation.
%   FE_HMM2D_PARA solves the parabolic equation
%              d/dt u =  div(a*grad(u))  +  f  in Omega
%                   u = u_D                 on the Dirichlet boundary
%              d/dn u = g                   on the Neumann boundary
%   with a highly oscillatory tensor a(x)
%   on a geometry described by triangles and parallelograms, respectively
%   and presents the solution graphically.
%
%   Time stepping is done using an implicit Euler method.
%
%
%   The code is available at http://anmc.epfl.ch/ and described in 
%   further detail in 
%
%   A. Abdulle and A. Nonnenmacher
%   "A short and versatile finite element multiscale code for
%   homogenization problems"
%   Computer Methods in Applied Mechanics and Engineering,
%   http://dx.doi.org/10.1016/j.cma.2009.03.019
%
%   Please cite this article in any publication describing research
%   performed using the software.
%
%
%
%   The structure of the directories is as follows:
%   fe-hmm          
%   +---examples
%   | +---elliptic    contains main file and files for the problem specific setup
%   | | +---data      contains geometry data files
%   | +---parabolic   contains main file and files for the problem specific setup  
%   | | +---data      contains geometry data files
%   | +---tools       contains problem independent algorithm files
%   
%   To adapt the program to a given setup of the equation, the user has to 
%   modify the geometry data files in ./data and the M-files
%   <f_t.m>, <g_t.m>, <u_d_t.m> and <tensor_a.m>. They have to be in the same
%   directory as <fe_hmm2d_para.m>.
%
%
%   This code is based in part on
%   [1] J. Alberty, C. Carstensen & S. Funken
%       "Remarks around 50 lines of Matlab: short finite element implementation",
%        Numerical Algorithms, Springer, 1999, 20, 117-137
%
%   Email           : assyr.abdulle@epfl.ch and achim.nonnenmacher@epfl.ch
%   Last updated    : 04/30/2010 with MATLAB 7.8
%
%   FE_HMM2D is Copyright (C) 2010 A. Abdulle and A. Nonnenmacher. 
%   The software is provided free for non-commercial use unter the terms of 
%   the GNU General Public License. See "copyright.m" for full details.   
%

%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


% check if tools directory was added to the path
if(exist('shapefunction_quad_lin.m')~=2)
   error('HMMFEM:path', strcat('Please add "tools" directory to the Matlab path.\n',...
       'You can do that with the command:\n',...
       'addpath(''../../tools'');'));
end


% Variables
epsilon =5e-2;
NMicro  =4;

delta= epsilon;


% Initialisation
Coordinates=load('./data/coordinates.dat');
Elements3  =load('./data/elements3.dat');
Elements4  =load('./data/elements4.dat');
Neumann    =load('./data/neumann.dat');
Dirichlet  =load('./data/dirichlet.dat');

T = 1; dt = 0.1; N = T/dt;
U = zeros(size(Coordinates,1),N+1);

% Initial Condition
U(:,1) = load('./data/u_0.dat');

bctype='periodic';
% bctype='dirichlet';

FreeNodes=setdiff(1:size(Coordinates,1),unique(Dirichlet));
A = sparse(size(Coordinates,1),size(Coordinates,1));
B = sparse(size(Coordinates,1),size(Coordinates,1));
b = zeros(size(Coordinates,1),1);

% HMM initialization, microgrid with quadrilaterals, periodic constraints
MicroElements=micromesh_elements(NMicro);
Constraints=make_constraints(NMicro, bctype);

% Assembly
% For meshes with many DOF this can be very slow due to Matlab's way of 
% handling sparse matrices. An optimized code can be found at the very
% bottom of this file.
for j=1:size(Elements3,1)
    M=hmm_stima_tri(NMicro,epsilon,MicroElements,...
        Coordinates(Elements3(j,:),:),Constraints, delta, bctype);
    A(Elements3(j,:), Elements3(j,:)) = A(Elements3(j,:), Elements3(j,:))+M;
end

for j=1:size(Elements4,1)
    M=hmm_stima_quad(NMicro,epsilon,MicroElements,...
        Coordinates(Elements4(j,:),:),Constraints, delta, bctype);
    A(Elements4(j,:), Elements4(j,:)) = A(Elements4(j,:), Elements4(j,:))+M;
end

% Assembly mass matrix
for j = 1:size(Elements3,1)
  B(Elements3(j,:),Elements3(j,:)) = B(Elements3(j,:),Elements3(j,:)) ...
      + det([1,1,1;Coordinates(Elements3(j,:),:)'])*[2,1,1;1,2,1;1,1,2]/24;
end

for j = 1:size(Elements4,1)
  B(Elements4(j,:),Elements4(j,:)) = B(Elements4(j,:),Elements4(j,:)) ...
      + det([1,1,1;Coordinates(Elements4(j,1:3),:)'])*...
      ([4 2 1 2; 2 4 2 1; 1 2 4 2; 2 1 2 4 ])/36;
end


% time steps
for n = 2:N+1
    b = sparse(size(Coordinates,1),1);

    % Volume Forces
    for j=1:size(Elements3,1)
        b(Elements3(j,:)) = b(Elements3(j,:))+ ...
            dt*det([1,1,1; Coordinates(Elements3(j,:),:)'])* ...
            f_t(sum(Coordinates(Elements3(j,:),:))/3, (n-1)*dt)/6;
    end

    for j=1:size(Elements4,1)
        b(Elements4(j,:)) = b(Elements4(j,:))+ ...
            dt*det([1,1,1; Coordinates(Elements4(j,1:3),:)'])* ...
            f_t(sum(Coordinates(Elements4(j,:),:))/4, (n-1)*dt)/4;
    end


    % Neumann conditions
    for j = 1 : size(Neumann,1)
        b(Neumann(j,:))=b(Neumann(j,:)) + dt*norm(Coordinates(Neumann(j,1),:)- ...
            Coordinates(Neumann(j,2),:)) * g_t(sum(Coordinates(Neumann(j,:),:))/2, (n-1)*dt)/2;
    end

    % previous timestep
    b = b + B * U(:,n-1);

    % Dirichlet conditions
    u = sparse(size(Coordinates,1),1);
    u(unique(Dirichlet)) = u_d_t(Coordinates(unique(Dirichlet),:),(n-1)*dt);
    b = b - (dt * A + B) * u;

    % Computation of the solution
    u(FreeNodes) = (dt*A(FreeNodes,FreeNodes)+ ...
        B(FreeNodes,FreeNodes))\b(FreeNodes);
    U(:,n) = u;
end

show_parabolic(Elements3,Elements4,Coordinates,U,N);





enorm=sqrt(U(:,N+1)'*A*U(:,N+1));
disp(sprintf('energy norm is |u(1)|=%g',enorm));
disp(sprintf('max         is |u(1)|=%g',max(U(:,N+1))));





%
% Appendix
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


% % Assembly - faster version
% % For meshes with many DOF an optimized way of assembling the stiffness
% % matrix should be used, such as:
%
% indexi=zeros(3*3*size(Elements3,1),1);
% indexj=zeros(3*3*size(Elements3,1),1);
% value= zeros(3*3*size(Elements3,1),1);
% 
% for j=1:size(Elements3,1)
%     disp(j);
%     M=hmm_stima_tri(NMicro,epsilon,MicroElements,...
%         Coordinates(Elements3(j,:),:),Constraints, delta, bctype);
%     
%     tmpi=repmat(Elements3(j,:),1,3);
%     tmpj=repmat(Elements3(j,:),3,1);
% 
%     indexi(1+(j-1)*9:j*9)=tmpi(:);
%     indexj(1+(j-1)*9:j*9)=tmpj(:);
%     value(1+(j-1)*9:j*9)=M(:);
%     
%     %A(Elements3(j,:), Elements3(j,:)) = A(Elements3(j,:), Elements3(j,:))+M;
% end
% A=sparse(indexi,indexj,value,...
%             size(Coordinates,1),size(Coordinates,1));
% 
%         
%         
% indexi=zeros(4*4*size(Elements4,1),1);
% indexj=zeros(4*4*size(Elements4,1),1);
% value= zeros(4*4*size(Elements4,1),1);
% 
% for j=1:size(Elements4,1)
%     disp(j);    
%     M=hmm_stima_quad(NMicro,epsilon,MicroElements,...
%         Coordinates(Elements4(j,:),:),Constraints, delta, bctype);
%     
%     tmpi=repmat(Elements4(j,:),1,4);
%     tmpj=repmat(Elements4(j,:),4,1);
% 
%     indexi(1+(j-1)*16:j*16)=tmpi(:);
%     indexj(1+(j-1)*16:j*16)=tmpj(:);
%     value(1+(j-1)*16:j*16)=M(:);
% end
% A=A+sparse(indexi,indexj,value,...
%             size(Coordinates,1),size(Coordinates,1));    