%FE_HMM2D_PARA_NONLIN_PARA   two-dimensional heterogeneous multiscale finite element method for the nonlinear heat equation.
%   FE_HMM2D_PARA_NONLIN_PARA solves the nonlinear parabolic equation
%              d/dt u =  div(a(u)*grad(u))  +  f  in Omega
%                   u = u_D                 on the Dirichlet boundary
%              d/dn u = g                   on the Neumann boundary
%   with a highly oscillatory tensor a(x,u) depending nonlinearly on u,
%   on a geometry described by triangles and parallelograms, respectively
%   and presents the solution graphically.
%
%   Time integration is done using an implicit Euler method implemented
%   using the Newton method.
%
%   A parallel implemention of the stiffness matrix assembling is available
%   (use the matlab command matlabpool to start)
%
%   The code is available at http://iacs.epfl.ch/anmc/ and described in 
%   further detail in 
%
%   A. Abdulle and G. Vilmart
%   "Fully discrete analysis of the finite element heterogeneous multiscale method 
%   for nonmonotone elliptic homogenization problems", 
%   submitted for publication. 32 pages.
%   Preprint available at: http://infoscience.epfl.ch/record/163326/
%
%   Please cite this article in any publication describing research
%   performed using the software.
%
%   The structure of the directories is as follows:
%   fe-hmm          
%   +---examples
%   | +---elliptic_nonlin   contains main file and files for the problem specific setup
%   | | +---data            contains geometry data files
%   | +---parabolic_nonlin  contains main file and files for the problem specific setup  
%   | | +---data            contains geometry data files
%   | +---tools             contains problem independent algorithm files
%   
%   To adapt the program to a given setup of the equation, the user has to 
%   modify the geometry data files in ./data and the M-files
%   <f.m>, <g.m>, <u_d.m> and <tensor_a_nonlin.m>. They have to be in the same
%   directory as <fe_hmm2d_nonlin.m>.
%
%   This code is a nonlinear extension of the linear code
%   FE_HMM2D, Copyright (C) 2009 A. Abdulle and A. Nonnenmacher described in 

%   A. Abdulle and A. Nonnenmacher
%   "A short and versatile finite element multiscale code for
%   homogenization problems"
%   Computer Methods in Applied Mechanics and Engineering,
%   Vol. 198 (2009) p. 2839-2859.
%
%   This code is based in part on
%   [1] J. Alberty, C. Carstensen & S. Funken
%       "Remarks around 50 lines of Matlab: short finite element implementation",
%        Numerical Algorithms, Springer, 1999, 20, 117-137
%
%   Email           : assyr.abdulle@epfl.ch and gilles.vilmart@epfl.ch
%   Last updated    : 08/07/2011 with MATLAB 7.9
%
%   FE_HMM2D_NONLIN is Copyright (C) 2011 A. Abdulle and G. Vilmart. 
%   The software is provided free for non-commercial use unter the terms of 
%   the GNU General Public License. See "copyright.m" for full details.   
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Parallel version: type Matlabpool 
if matlabpool('size')==0
    disp('Warning: First use the command matlabpool to start the parallel computing pool.')
end

% check if tools directory was added to the path
if(exist('hmm_microstima_nonlin.m')~=2)
   error('HMMFEM:path', strcat('Please add "tools" directory to the Matlab path.\n',...
       'You can do that with the command:\n',...
       'addpath(''../../tools'');'));
end

% check if FE-HMM has been installed
if(exist('shapefunction_quad_lin.m')~=2)
   error('HMMFEM:path', strcat('To use FE_HMM2D_NONLIN, please install first FE_HMM2D'));
end


% Variables
epsilon =5e-2;

NMicro  =4;

delta= epsilon;

% tolerance for Newton iterations
tol=1e-08;

% Initialisation
Coordinates=load('./data/coordinates.dat');
Elements3  =load('./data/elements3.dat');
Elements4  =load('./data/elements4.dat');
Neumann    =load('./data/neumann.dat');
Dirichlet  =load('./data/dirichlet.dat');

T = 1.; dt = 0.1; N = T/dt;

U = zeros(size(Coordinates,1),N+1);

% Initial Condition
U(:,1) = load('./data/u_0.dat');
u=U(:,1);

tic

bctype='periodic';
% bctype='dirichlet';

FreeNodes=setdiff(1:size(Coordinates,1),unique(Dirichlet));
B = sparse(size(Coordinates,1),size(Coordinates,1));
b = sparse(size(Coordinates,1),1);

% HMM initialization, microgrid with quadrilaterals, periodic constraints
MicroElements=micromesh_elements(NMicro);
Constraints=make_constraints(NMicro, bctype);


% Assembly mass matrix
for j = 1:size(Elements3,1)
  B(Elements3(j,:),Elements3(j,:)) = B(Elements3(j,:),Elements3(j,:)) ...
      + det([1,1,1;Coordinates(Elements3(j,:),:)'])*[2,1,1;1,2,1;1,1,2]/24;
end

for j = 1:size(Elements4,1)
  B(Elements4(j,:),Elements4(j,:)) = B(Elements4(j,:),Elements4(j,:)) ...
      + det([1,1,1;Coordinates(Elements4(j,1:3),:)'])*...
      ([4 2 1 2; 2 4 2 1; 1 2 4 2; 2 1 2 4 ])/36;
end

for n = 1:N
    disp(sprintf('timestep %g over %g...',n,N));
    b = sparse(size(Coordinates,1),1);
    % Volume Forces
    for j=1:size(Elements3,1)
        b(Elements3(j,:)) = b(Elements3(j,:))+ ...
            det([1,1,1; Coordinates(Elements3(j,:),:)'])* ...
            f_t(sum(Coordinates(Elements3(j,:),:))/3, n*dt)/6;
    end
    for j=1:size(Elements4,1)
        b(Elements4(j,:)) = b(Elements4(j,:))+ ...
            det([1,1,1; Coordinates(Elements4(j,1:3),:)'])* ...
            f_t(sum(Coordinates(Elements4(j,:),:))/4, n*dt)/4;
    end
    % Neumann conditions
    for j = 1 : size(Neumann,1)
        b(Neumann(j,:))=b(Neumann(j,:)) + norm(Coordinates(Neumann(j,1),:)- ...
            Coordinates(Neumann(j,2),:)) * g_t(sum(Coordinates(Neumann(j,:),:))/2, n*dt)/2;
    end
    c =  dt * b + B * U(:,n);
        
    % Newton iterations for computing the solution
    k=0;
    incr0=1;
    while incr0>tol
    k=k+1;
    uold=u;
        
    if matlabpool('size')==0
    %disp('NON PARALLEL COMPUTING');
    A = sparse(size(Coordinates,1),size(Coordinates,1));
    % matrix Ajac for Newton iterations
    Ajac = sparse(size(Coordinates,1),size(Coordinates,1));
    for j=1:size(Elements3,1)
        [MA,MB]=hmm_stima_tri_nonlin_jac(u(Elements3(j,:)),NMicro,NMicro,epsilon,...
            MicroElements,MicroElements,Coordinates(Elements3(j,:),:),Constraints,Constraints,delta, bctype);
        A(Elements3(j,:), Elements3(j,:)) = A(Elements3(j,:), Elements3(j,:))+MA;
        Ajac(Elements3(j,:), Elements3(j,:)) = Ajac(Elements3(j,:), Elements3(j,:))+MB;
    end
    for j=1:size(Elements4,1)
        [MA,MB]=hmm_stima_quad_nonlin_jac(u(Elements4(j,:)),NMicro,NMicro,epsilon,...
            MicroElements,MicroElements,Coordinates(Elements4(j,:),:),Constraints,Constraints,delta, bctype);
        A(Elements4(j,:), Elements4(j,:)) = A(Elements4(j,:), Elements4(j,:))+MA;
        Ajac(Elements4(j,:), Elements4(j,:)) = Ajac(Elements4(j,:), Elements4(j,:))+MB;
    end
    else
    %disp('PARALLEL COMPUTING');
    valuepar=[];
    parfor j=1:size(Elements3,1)
    [MA,MB]=hmm_stima_tri_nonlin_jac(u(Elements3(j,:)),NMicro,NMicro,epsilon,...
            MicroElements,MicroElements,Coordinates(Elements3(j,:),:),Constraints,Constraints,delta, bctype);
    tmpi=repmat(Elements3(j,:),1,3);
    tmpj=repmat(Elements3(j,:),3,1);
    newpar=[tmpi(:),tmpj(:),MA(:),MB(:)]';
    valuepar=[valuepar,newpar];
    end
    parfor j=1:size(Elements4,1)
    [MA,MB]=hmm_stima_quad_nonlin_jac(u(Elements4(j,:)),NMicro,NMicro,epsilon,...
            MicroElements,MicroElements,Coordinates(Elements4(j,:),:),Constraints,Constraints,delta, bctype);
    tmpi=repmat(Elements4(j,:),1,4);
    tmpj=repmat(Elements4(j,:),4,1);
    newpar=[tmpi(:),tmpj(:),MA(:),MB(:)]';
    valuepar=[valuepar,newpar];
    end
    A=sparse(valuepar(1,:),valuepar(2,:),valuepar(3,:),size(Coordinates,1),size(Coordinates,1)); 
    Ajac=sparse(valuepar(1,:),valuepar(2,:),valuepar(4,:),size(Coordinates,1),size(Coordinates,1)); 
    end
    
    ff=-(dt*A+B)*u+c;
    mat=dt*(A+Ajac)+B;
    u(FreeNodes) = u(FreeNodes)+mat(FreeNodes,FreeNodes)\ff(FreeNodes);
    incr0=norm(u-uold,inf);
    end
    
    U(:,n+1) = u;
    
end
toc


show_parabolic(Elements3,Elements4,Coordinates,U,N);

enorm=sqrt(U(:,N+1)'*A*U(:,N+1));
disp(sprintf('energy norm is |u(1)|=%g',enorm));
disp(sprintf('max         is |u(1)|=%g',max(U(:,N+1))));
