function M = hmm_microstima_quad(vertices,...
    MacroQuadNode, epsilon)
%HMM_MICROSTIMA_QUAD   Computes element stiffness matrix for micro parallelograms.
%   M = HMM_MICROSTIMA_QUAD(VERTICES, MACROQUADNODE, EPSILON) 
%   computes the entry to the micro element stiffness matrix for parallelograms.
%
%   VERTICES has dimension 4 x 2, where the first column gives 
%   the x-coordinates of the vertices and the second column their 
%   y-coordinates. 
%
%   MACROQUADNODE has dimension 1 x 2 and contains the x and y
%   coordinates of the current quadrature node of the macro element
%
%   EPSILON is the periodicity of the tensor a
%   
%   M has dimension 4 x 4. The vertices are numbered 
%   anti-clockwise. 
%
%
%   This function should not be modified.
%
%
%   The code is available at http://anmc.epfl.ch/ and described in 
%   further detail in 
%
%   A. Abdulle and A. Nonnenmacher
%   "A short and versatile finite element multiscale code for
%   homogenization problems"
%   Computer Methods in Applied Mechanics and Engineering,
%   http://dx.doi.org/10.1016/j.cma.2009.03.019
%
%   Please cite this article in any publication describing research
%   performed using the software.
%
%
%   Email           : assyr.abdulle@epfl.ch and achim.nonnenmacher@epfl.ch
%   Last updated    : 04/29/2009 with MATLAB 7.4
%
%   FE_HMM2D is Copyright (C) 2009 A. Abdulle and A. Nonnenmacher. 
%   The software is provided free for non-commercial use unter the terms of 
%   the GNU General Public License. See "copyright.m" for full details.   

%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


% map from reference element
Map=[vertices(2,:)-vertices(1,:);vertices(4,:)-vertices(1,:)]';

% gauss quadrature at the 4 quadrature nodes, reference element and mapped
% element
quadnodes_ref=[.5-sqrt(3)/6, .5+sqrt(3)/6, .5-sqrt(3)/6, .5+sqrt(3)/6;...
               .5-sqrt(3)/6, .5-sqrt(3)/6, .5+sqrt(3)/6, .5+sqrt(3)/6];
quadnodes= Map*quadnodes_ref+repmat(vertices(1,:)',1,4);

M=zeros(4,4);
for node=1:4
    quadnode=quadnodes(:,node);
    x=quadnodes_ref(1,node);
    y=quadnodes_ref(2,node);
    
    phi_prime_hat=[y-1, x-1; 1-y, -x; y, x; -y, 1-x];
    phi_prime_invD=(phi_prime_hat/Map);    
    
    % evaluate coefficiency tensor at the specific nodes
    a=tensor_a(MacroQuadNode, quadnode/epsilon);
    M=M+(det(Map))*phi_prime_invD*a*phi_prime_invD';
end
M=M/4;

end