function [A_macro]=hmm_stima_quad(NMicro, epsilon, MicroElements,...
    MacroVertices, Constraints, delta, bctype)
%HMM_STIMA_QUAD   Computes element stiffness matrix for macro parallelograms.
%   A_MACRO = HMM_STIMA_QUAD(VERTICES, MACROQUADNODE, EPSILON) 
%   computes the entry to the macro element stiffness matrix for macro
%   parallelograms using parallelograms on the micro sampling domain and
%   specified boundary conditions on the micro problem.
%
%   NMICRO  is the number of degrees of freedom per space dimension in the
%   micro domain
%
%   EPSILON is the periodicity of the tensor a.
%
%   MICROELEMENTS has dimension (NMicro-1)^2 x 4 and contain the node
%   numbers of the triangulation of the micro sampling domain.
%
%   MACROVERTICES contains the node coordinates of the macro parallelograms.
%
%   CONSTRAINTS contains information about the micro boundary problem. The
%   format depends on whether dirichlet or periodic boundary conditions are
%   used. See make_contraints.m for details.
%
%   DELTA is the width or height of the square sampling domain.
%
%   BCTYPE is either "dirichlet" or "periodic" and determines the boundary
%   condition of the micro problem.
%   
%   A_MACRO has dimension 4 x 4. The vertices are numbered 
%   anti-clockwise. 
%
%
%   This function should not be modified.
%
%
%   The code is available at http://anmc.epfl.ch/ and described in 
%   further detail in 
%
%   A. Abdulle and A. Nonnenmacher
%   "A short and versatile finite element multiscale code for
%   homogenization problems"
%   Computer Methods in Applied Mechanics and Engineering,
%   http://dx.doi.org/10.1016/j.cma.2009.03.019
%
%   Please cite this article in any publication describing research
%   performed using the software.
%
%
%   Email           : assyr.abdulle@epfl.ch and achim.nonnenmacher@epfl.ch
%   Last updated    : 04/29/2009 with MATLAB 7.4
%
%   FE_HMM2D is Copyright (C) 2009 A. Abdulle and A. Nonnenmacher. 
%   The software is provided free for non-commercial use unter the terms of 
%   the GNU General Public License. See "copyright.m" for full details.   

%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% initialization of variables
A_macro=zeros(4,4);

% transformation matrix for affine map to reference element
Map=[MacroVertices(2,:)-MacroVertices(1,:);MacroVertices(4,:)-MacroVertices(1,:)]';
% reference and physical quadrilateral quadrature nodes
quadnodes_ref=[.5-sqrt(3)/6, .5+sqrt(3)/6, .5-sqrt(3)/6, .5+sqrt(3)/6;...
               .5-sqrt(3)/6, .5-sqrt(3)/6, .5+sqrt(3)/6, .5+sqrt(3)/6];
quadnodes= Map*quadnodes_ref+repmat(MacroVertices(1,:)',1,4);

for node_no=1:4
    % select current quadrature nodes
    quadnode=quadnodes(:,node_no);
    refquadnode=quadnodes_ref(:,node_no);

    % Generate micro-coordinates
    [MicroCoordinates]=micromesh_coords(NMicro, quadnode, delta);

    % Initialization
    NoOfNodes = size(MicroCoordinates,1);
    A = sparse(NoOfNodes,NoOfNodes);

    %Stiffness matrix Assembly
    % For meshes with many DOF this can be very slow due to Matlab's way of 
    % handling sparse matrices. An optimized code can be found at the very
    % bottom of this file.    
    for j=1:size(MicroElements,1)
        A(MicroElements(j,:), MicroElements(j,:)) = ...
            A(MicroElements(j,:), MicroElements(j,:)) + ...
            hmm_microstima_quad(MicroCoordinates(MicroElements(j,:),:),...
            quadnode, epsilon);
    end
        

    % Get constraint matrix and corresponding rhs for the lagrange multiplier
    % micro boundary conditions.    
    switch(lower(bctype))
        case{'dirichlet'}
            [ConstraintMat, ConstraintRhs]=...
                micro_constraints_dirichlet(Constraints, Map, MacroVertices ,...
                MicroElements, MicroCoordinates, refquadnode);
        case{'periodic'}
            [ConstraintMat, ConstraintRhs]=...
                micro_constraints_periodic(Constraints, Map, MacroVertices ,...
                MicroElements, MicroCoordinates, refquadnode);
    end

    % Assemble matrix AConstr for constrained system and corresponding rhs
    AConstr=[A ConstraintMat';...
        ConstraintMat sparse(size(ConstraintMat,1),size(ConstraintMat,1))];
    RhsConstr=[zeros(NoOfNodes,4);ConstraintRhs];

    % solve constrained microproblem
    x=AConstr\RhsConstr;
    alpha=x(1:size(MicroCoordinates,1),:);

    % area of macro-element and micro-domain
    K_macro=det([1,1,1; MacroVertices(1:3,:)']);
    K_micro=delta^2;

    %The contribution to the macro stiffness matrix
    A_macro=A_macro+ .25 * K_macro/K_micro *alpha'*A*alpha;
end

end







%
% Appendix
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%     % stiffness matrix assembly - fast version
%
%     indexx=zeros(4*4*size(MicroElements,1),1);
%     indexy=zeros(4*4*size(MicroElements,1),1);
%     value= zeros(4*4*size(MicroElements,1),1);
% 
%     for j=1:size(MicroElements,1)
%         M=hmm_microstima_quad(MicroCoordinates(MicroElements(j,:),:),...
%              quadnode, epsilon);
% 
%         tmpa=repmat(MicroElements(j,:),1,4);
%         tmpb=repmat(MicroElements(j,:),4,1);
%         tmpM(:,:)=M;
%         indexx(1+(j-1)*16:j*16)=tmpa(:);
%         indexy(1+(j-1)*16:j*16)=tmpb(:);
%         value(1+(j-1)*16:j*16)=tmpM(:);
%     end
%     A=sparse(indexx,indexy,value,...
%         size(MicroCoordinates,1),size(MicroCoordinates,1));
