function [A_macro]=hmm_stima_tri(NMicro, epsilon, MicroElements,...
    MacroVertices, Constraints, delta, bctype)
%HMM_STIMA_TRI   Computes element stiffness matrix for macro triangles.
%   A_MACRO = HMM_STIMA_TRI(VERTICES, MACROQUADNODE, EPSILON) 
%   computes the entry to the macro element stiffness matrix for macro
%   triangles using parallelograms on the micro sampling domain and
%   specified boundary conditions on the micro problem.
%
%   NMICRO  is the number of degrees of freedom per space dimension in the
%   micro domain
%
%   EPSILON is the periodicity of the tensor a.
%
%   MICROELEMENTS has dimension (NMicro-1)^2 x 4 and contain the node
%   numbers of the triangulation of the micro sampling domain.
%
%   MACROVERTICES contains the node coordinates of the macro triangles.
%
%   CONSTRAINTS contains information about the micro boundary problem. The
%   format depends on whether dirichlet or periodic boundary conditions are
%   used. See make_contraints.m for details.
%
%   DELTA is the width or height of the square sampling domain.
%
%   BCTYPE is either "dirichlet" or "periodic" and determines the boundary
%   condition of the micro problem.
%   
%   A_MACRO has dimension 4 x 4. The vertices are numbered 
%   anti-clockwise. 
%
%   This function should not be modified.
%
%
%   The code is available at http://anmc.epfl.ch/ and described in 
%   further detail in 
%
%   A. Abdulle and A. Nonnenmacher
%   "A short and versatile finite element multiscale code for
%   homogenization problems"
%   Computer Methods in Applied Mechanics and Engineering,
%   http://dx.doi.org/10.1016/j.cma.2009.03.019
%
%   Please cite this article in any publication describing research
%   performed using the software.
%
%
%   Email           : assyr.abdulle@epfl.ch and achim.nonnenmacher@epfl.ch
%   Last updated    : 04/29/2009 with MATLAB 7.4
%
%   FE_HMM2D is Copyright (C) 2009 A. Abdulle and A. Nonnenmacher. 
%   The software is provided free for non-commercial use unter the terms of 
%   the GNU General Public License. See "copyright.m" for full details.   

%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


% generate micro-coordinates
bary=sum(MacroVertices(:,:))/3;
[MicroCoordinates]=micromesh_coords(NMicro, bary, delta);

% initialization of variables
NoOfNodes = size(MicroCoordinates,1);
A = sparse(NoOfNodes,NoOfNodes);

% stiffness matrix assembly
% For meshes with many DOF this can be very slow due to Matlab's way of 
% handling sparse matrices. An optimized code can be found at the very
% bottom of this file.
for j=1:size(MicroElements,1)
    A(MicroElements(j,:), MicroElements(j,:)) = ...
        A(MicroElements(j,:), MicroElements(j,:)) + ...
        hmm_microstima_quad(MicroCoordinates(MicroElements(j,:),:),...
            bary, epsilon);
end

        
        

% solution of cell problems dilatation for the macro element
D=[MacroVertices(2,:)-MacroVertices(1,:);MacroVertices(3,:)-MacroVertices(1,:)]';

% get constraint matrix and corresponding rhs for the lagrange multiplier

switch(lower(bctype))
    case{'dirichlet'}
        [ConstraintMat, ConstraintRhs]=...
            micro_constraints_dirichlet(Constraints, D, MacroVertices ,...
            MicroElements, MicroCoordinates, 0);
    case{'periodic'}
        [ConstraintMat, ConstraintRhs]=...
            micro_constraints_periodic(Constraints, D, MacroVertices ,...
            MicroElements, MicroCoordinates, 0);
end


% assemble matrix for constraint system and corresponding rhs
AConstr=[A ConstraintMat';...
    ConstraintMat sparse(size(ConstraintMat,1),size(ConstraintMat,1))];
RhsConstr=[zeros(NoOfNodes,3);ConstraintRhs];

% finally solve
x=AConstr\RhsConstr;
alpha=x(1:size(MicroCoordinates,1),:);

% area of macro-element and micro-domain
K_macro=.5*det([1,1,1; MacroVertices']);
K_micro=delta^2;

% the entries for the macro stiffness matrix
A_macro=K_macro/K_micro *alpha'*A*alpha;

end




%
% Appendix
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% % stiffness matrix assembly - fast version
% indexx=zeros(4*4*size(MicroElements,1),1);
% indexy=zeros(4*4*size(MicroElements,1),1);
% value= zeros(4*4*size(MicroElements,1),1);
% 
% for j=1:size(MicroElements,1)
%     M=hmm_microstima_quad(MicroCoordinates(MicroElements(j,:),:),...
%             bary, epsilon);
% 
%     tmpa=repmat(MicroElements(j,:),1,4);
%     tmpb=repmat(MicroElements(j,:),4,1);
%     tmpM(:,:)=M;
%     indexx(1+(j-1)*16:j*16)=tmpa(:);
%     indexy(1+(j-1)*16:j*16)=tmpb(:);
%     value(1+(j-1)*16:j*16)=tmpM(:);
% end
% A=sparse(indexx,indexy,value,...
%             size(MicroCoordinates,1),size(MicroCoordinates,1));
