function [Constraints]=make_constraints(N, bctype)
%MAKE_CONSTRAINTS   Computes constraint data for micro problem.
%   CONSTRAINTS = MAKE_CONSTRAINTS(N, BCTYPE) 
%   depending on BCTYPE, 
%   - for periodic boundary conditions, node numbers for identified nodes are
%     stored in every row of the matrix CONSTRAINTS
%   - for dirichlet boundary conditions, the node numbers on the boundary 
%     which will be set to zero are stored in one vector CONSTRAINTS
%
%   N=NMICRO  is the number of degrees of freedom per space dimension in the
%   micro domain
%
%   BCTYPE is either "dirichlet" or "periodic" and determines the boundary
%   condition of the micro problem.
%   
%   CONSTRAINTS contains information about the micro boundary problem. The
%   format depends on whether dirichlet or periodic boundary conditions are
%   used. 
%
%
%   This function should not be modified.
%
%
%   The code is available at http://anmc.epfl.ch/ and described in 
%   further detail in 
%
%   A. Abdulle and A. Nonnenmacher
%   "A short and versatile finite element multiscale code for
%   homogenization problems"
%   Computer Methods in Applied Mechanics and Engineering,
%   http://dx.doi.org/10.1016/j.cma.2009.03.019
%
%   Please cite this article in any publication describing research
%   performed using the software.
%
%
%   Email           : assyr.abdulle@epfl.ch and achim.nonnenmacher@epfl.ch
%   Last updated    : 04/29/2009 with MATLAB 7.4
%
%   FE_HMM2D is Copyright (C) 2009 A. Abdulle and A. Nonnenmacher. 
%   The software is provided free for non-commercial use unter the terms of 
%   the GNU General Public License. See "copyright.m" for full details.   

%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


switch(lower(bctype))
    case{'periodic'}
        % Build periodic boundary condition matrix
        % PerConstraints(i,1) and PerConstraints(j,2) specify the nodes that are
        % periodic to each other, leaving out the one constraint that are implicitly
        % defined by the others already
        Constraints(:,1)=[1:N 1:N:N*N-N];
        Constraints(:,2)=[(N-1)*N+1:N*N N:N:N*N-N];
        
    case{'dirichlet'}
        % Constraints will contain the boundary nodes
        % bottom, top, left, right (not counting same node twice)
        Constraints=[1:N N*N-N+1:N*N 1+N:N:N*N-2*N+1 2*N:N:N*N-N];

end