function [ConstraintMat, ConstraintRhs ]=...
    micro_constraints_dirichlet(Constraints, MacroMap, MacroVertices,...
    MicroElements, MicroCoordinates, refquadnode)
%MICRO_CONSTRAINTS_DIRICHLET   Computes micro Dirichlet boundary conditions matrix
%   [CONSTRAINTMAT, CONSTRAINTRHS] = MICRO_CONSTRAINTS_DIRICHLET(...
%             CONSTRAINTS, MACROMAP, MACROVERTICES, MICROELEMENTS, ...
%             MICROCOORDINATES, REFQUADNODE) 
%   computes a matrix and corresponding vector used to enforce Dirichlet 
%   boundary conditions on the micro problem. To do this, in hmm_stima_quad
%   and hmm_stima_tri, respectively the matrix A_Micro will be augmented by
%   CONSTRAINTMAT and boundary conditions are enforced using Langrange
%   multipliers.
%
%   CONSTRAINTS contains information about the micro boundary problem. 
%   See make_contraints.m for details.
%
%   MACROMAP is the transformation matrix for the affine map to the 
%   reference element.
%
%   MACROVERTICES contains the node coordinates of the macro elements.
%
%   MICROELEMENTS has dimension (NMicro-1)^2 x 4 and contain the node
%   numbers of the triangulation of the micro sampling domain.
%
%   MICROCOORDINATES contains the coordinates of the nodes
%   of the triangulation of the micro sampling domain.
%
%   REFQUADNODES contains the quadrature nodes in the reference square
%
%   CONSTRAINTMAT contains the matrix that will be used to enforce boundary
%   conditions via Lagrange multipliers. This matrix together with the
%   matrix A_MICRO will be used for the augmented matrix in hmm_stima_quad
%   and hmm_stima_tri, respectively.
%
%   CONSTRAINTRHS contains the corresponding right hand side.
%
%
%   This function should not be modified.
%
%
%   The code is available at http://anmc.epfl.ch/ and described in 
%   further detail in 
%
%   A. Abdulle and A. Nonnenmacher
%   "A short and versatile finite element multiscale code for
%   homogenization problems"
%   Computer Methods in Applied Mechanics and Engineering,
%   http://dx.doi.org/10.1016/j.cma.2009.03.019
%
%   Please cite this article in any publication describing research
%   performed using the software.
%
%
%   Email           : assyr.abdulle@epfl.ch and achim.nonnenmacher@epfl.ch
%   Last updated    : 04/29/2009 with MATLAB 7.4
%
%   FE_HMM2D is Copyright (C) 2009 A. Abdulle and A. Nonnenmacher. 
%   The software is provided free for non-commercial use unter the terms of 
%   the GNU General Public License. See "copyright.m" for full details.   

%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


% determine whether we have macro triangles or quadrilaterals
MacroNodes=size(MacroVertices, 1);

% Dirichlet Boundary conditions
NoOfDirichletNodes=size(Constraints,2);
ConstraintMat=sparse(1:NoOfDirichletNodes, ...
    Constraints, ones(NoOfDirichletNodes,1));

% Micro-coordinates of the dirichlet-nodes
x_nodes= MicroCoordinates(Constraints', :);
 
% coordinates in the macro reference triangle corresponding to the 
% micro-nodes with dirichlet boundary conditions
x_node_ref = (MacroMap\(x_nodes-repmat(MacroVertices(1,:), ...
    NoOfDirichletNodes,1))')';

% corresponding rhs
if (MacroNodes==3) % we have macro triangles
    ConstraintRhs=shapefunction_tri(x_node_ref);
end
if (MacroNodes==4) % we have macro quadrilaterals
    ConstraintRhs=shapefunction_quad_lin(x_node_ref, refquadnode);
end

end