function [ConstraintMat, ConstraintRhs ]=...
    micro_constraints_periodic(Constraints, MacroMap, MacroVertices,...
    MicroElements, MicroCoordinates, refquadnode)
%MICRO_CONSTRAINTS_PERIODIC   Computes micro periodic boundary conditions matrix
%   [CONSTRAINTMAT, CONSTRAINTRHS] = MICRO_CONSTRAINTS_PERIODIC(...
%             CONSTRAINTS, MACROMAP, MACROVERTICES, MICROELEMENTS, ...
%             MICROCOORDINATES, REFQUADNODE) 
%   computes a matrix and corresponding vector used to enforce periodic
%   boundary conditions on the micro problem. To do this, in hmm_stima_quad
%   and hmm_stima_tri, respectively the matrix A_Micro will be augmented by
%   CONSTRAINTMAT and boundary conditions are enforced using Langrange
%   multipliers.
%
%   CONSTRAINTS contains information about the micro boundary problem. 
%   See make_contraints.m for details.
%
%   MACROMAP is the transformation matrix for the affine map to the 
%   reference element.
%
%   MACROVERTICES contains the node coordinates of the macro elements.
%
%   MICROELEMENTS has dimension (NMicro-1)^2 x 4 and contain the node
%   numbers of the triangulation of the micro sampling domain.
%
%   MICROCOORDINATES contains the coordinates of the nodes
%   of the triangulation of the micro sampling domain.
%
%   REFQUADNODES contains the quadrature nodes in the reference square
%
%   CONSTRAINTMAT contains the matrix that will be used to enforce boundary
%   conditions via Lagrange multipliers. This matrix together with the
%   matrix A_MICRO will be used for the augmented matrix in hmm_stima_quad
%   and hmm_stima_tri, respectively.
%
%   CONSTRAINTRHS contains the corresponding right hand side.
%
%
%   This function should not be modified.
%
%
%   The code is available at http://anmc.epfl.ch/ and described in 
%   further detail in 
%
%   A. Abdulle and A. Nonnenmacher
%   "A short and versatile finite element multiscale code for
%   homogenization problems"
%   Computer Methods in Applied Mechanics and Engineering,
%   http://dx.doi.org/10.1016/j.cma.2009.03.019
%
%   Please cite this article in any publication describing research
%   performed using the software.
%
%
%   Email           : assyr.abdulle@epfl.ch and achim.nonnenmacher@epfl.ch
%   Last updated    : 04/29/2010 with MATLAB 7.4
%
%   FE_HMM2D is Copyright (C) 2010 A. Abdulle and A. Nonnenmacher. 
%   The software is provided free for non-commercial use unter the terms of 
%   the GNU General Public License. See "copyright.m" for full details.   

%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


% determine whether we have macro triangles or quadrilaterals
MacroNodes=size(MacroVertices, 1);

% Constraint 1: 
% Periodic Boundary conditions
NoOfPerNodes=size(Constraints,1);
ConstraintMat=sparse(repmat(1:NoOfPerNodes,1,2), ...
    [Constraints(:,1) Constraints(:,2)], ...
    [ones(NoOfPerNodes,1); -ones(NoOfPerNodes,1)]);

% coordinates of periodic boundary conditions
% x_node1 and x_node2 are corresponding periodic nodes
x_node1= MicroCoordinates(Constraints(:,1), :);
x_node2= MicroCoordinates(Constraints(:,2), :);
 
% coordinates in the reference quadrilateral corresponding to the nodes with
% periodic boundary conditions
x_node_ref1 = (MacroMap\(x_node1-repmat(MacroVertices(1,:), NoOfPerNodes,1))')';
x_node_ref2 = (MacroMap\(x_node2-repmat(MacroVertices(1,:), NoOfPerNodes,1))')';

% corresponding rhs
if (MacroNodes==3) % we have macro triangles
    ConstraintRhs=shapefunction_tri(x_node_ref1)-...
        shapefunction_tri(x_node_ref2);
end
if (MacroNodes==4) % we have macro quadrilaterals
    ConstraintRhs=shapefunction_quad_lin(x_node_ref1, refquadnode)-...
        shapefunction_quad_lin(x_node_ref2, refquadnode);
end


% Constraint 2: 
% zero-average u^h-phi^H_i
NoOfNodes = size(MicroCoordinates,1);
b=zeros(NoOfNodes,1);

for j=1:size(MicroElements,1)
    b(MicroElements(j,:))=b(MicroElements(j,:))+...
        det([1,1,1; MicroCoordinates(MicroElements(j,1:3),:)'])/4;
end



% merge both to one unified matrix / vector
ConstraintMat=[b';ConstraintMat];
ConstraintRhs=[zeros(1,MacroNodes);ConstraintRhs];

end
