      subroutine skrock(neqn,nsto,t,nsteps,h,winc,y,f,g,
     &   genrnd,work,iwork,idid)
c ----------------------------------------------------------

c    Optimal first weak order explicit stabilized methods 
c    for stiff Ito stochastic differential equations 
c    dX(t) = f(t,X(t)) dt + sum_i g^i(t,X(t)) dW_i(t)
c
c    This algorithm is described in
c    [ ] Assyr Abdulle, Ibrahim Almuslimani, 
c        and Gilles Vilmart, Optimal explicit stabilized 
c        integrator of weak order one for stiff and ergodic 
c        stochastic differential equations, 
c        SIAM/ASA J. Uncertain. Quantif. 6 (2018), no. 2, 937–964.
c        https://doi.org/10.1137/17M1145859
c
c    Version of April 4, 2018
c
c    PLEASE CITE THE ABOVE PAPER WHEN USING SKROCK.
c ----------------------------------------------------------
c    The size (along the negative axis) of the stability 
c    domains increases quadratically with the stage number.
c
c    Intended for problems of large dimensions with
c    eigenvalues of the Jacobian close to the negative
c    real axis. Typically for problems originating from
c    parabolic PDEs.
c--------------------------------------------------------------------
c     Input parameters  
c     ----------------  
c     NEQN:       Number of differential equations of the system 
c                 (integer).
c
c     NSTO:       Number of independent Wiener processes.
c
c     T:          Initial point of integration (double precision).
c
c     NSTEPS:     Number of timesteps
c
c     H:          Constant timestep size
c
c     Y(NEQN):    Initial value of the solution 
c                 (double precision array of length neqn).
c
c     F:          Name (external) of subroutine computing the value 
c                 of f(x,y). Must have the form
c
c                   subroutine f(neqn,t,y,dy)
c                   double precision y(neqn),dy(neqn)
c                   integer neqn
c                   dy(1)=...
c                   ...
c                   dy(neqn)=...
c                   return
c                   end 
c
c     G:          Name (external) of subroutine computing the value 
c                 of g^i(x,y) where i is the noise index. 
c                 Must have the form
c
c                   subroutine g(neqn,nsto,i,t,y,dy)
c                   double precision y(neqn),dy(neqn)
c                   integer neqn
c                   dy(1)=...
c                   ...
c                   dy(neqn)=...
c                   return
c                   end 
c
c                 Implementation:   
c                 for stability issues when the problem 
c                 is originating from parabolic PDEs, transforming 
c                 inhomogeneous boundary conditions in homogeneous ones
c                 (by adding the appropriate function to the right-hand side)
c                 may increase the performance of the code.
c
c     RHO:        Name (external) of a function (double precision) 
c                 giving the spectral radius of the Jacobian 
c                 matrix  of f at (t,y). Must have the form
c                 
c                   double precision function rho(neqn,t,y)
c                   double precision y(neqn),t
c                   integer neqn
c                   ...
c                   rho=... 
c                   return
c                   end
c               
c                 N.b. Gerschgorin's theorem can be helpful. If the
c                 Jacobian is known to be constant it should be 
c                 specified by setting iwork(2)=1 (see below).
c
c                 ROCK2 can also compute this estimate. In that 
c                 case, provide a dummy function rho(neqn,t,y) and 
c                 set iwork(1)=0 (see below).
c
c                 If it is possible to give an estimate of 
c                 the spectral radius, it should be preferred to
c                 the estimate computed internally by ROCK2.
c
c     IWORK(*):   Integer array of length 15 that gives information 
c                 on how the problem is to be solved and communicates 
c                 statistics about the integration process.
c               
c     IWORK(1):   =0 SK-ROCK attempts to compute the spectral radius 
c                    internally. Define a dummy function
c
c                    double precision function rho(neqn,t,y)
c                    double precision y(neqn),t
c                    integer neqn
c                    rho=0.d0 
c                    return
c                    end
c
c                 =1 RHO returns an upper bound of the spectral 
c                    radius  of the Jacobian matrix of f at (t,y).
c    
c     IWORK(2):   =0 General non-commutative noise case                 
c                 =1 Diagonal noise case

c     IWORK(3)    =1 Postprocessed method, else normal method
c
c     IWORK(4):   =0 automatic choice of stage number
c                 otherwise force stage number to IWORK(4)
c
c     WORK(*) :     Workspace of length 12*neqn+nsto if iwork(2)=0,
c                   otherwise of length 12*neqn+neqn*nsto.
c                   Work(1),..,work(4*neqn) serve as
c                   working space for the solution of
c                   the ode.
c                   Work(4*neqn+1),..,work(5*neqn)
c                   serve as working space for the
c                   internal computation of the 
c                   spectral radius of the Jacobian.
c                  
c     IDID:         Report on successfulness upon return
c                   (integer).
c
c
c     Output parameters 
c     -----------------
c     T:          T-value for which the solution has been computed
c                 (after successful return t=tend).
c
c     Y(NEQN):    Numerical solution at tend.
c
c     IDID:       Reports what happened upon return
c
c     IDID        =1 Successful computation t=tend.
c                 =2 Successful computation of one step
c                    to continue call ROCK2 again without
c                    altering any arguments.
c                 =-1 Invalid input parameters.
c                 =-2 Stepsize becomes to small.
c                 =-3 The method used in ROCK2 to estimate 
c                     the spectral radius did not converge.
c
c     IWORK(5)    =Number of function evaluations.
c     IWORK(6)    =Number of steps.
c     IWORK(7)    =Number of accepted steps.
c     IWORK(8)    =Number of rejected steps.
c     IWORK(9)    =Number of evaluations of f used
c                  to estimate the spectral radius
c                  (equal to zero if iwork(1)=1).
c     IWORK(10)   =Maximum number of stages used.
c     IWORK(11)   =Maximum value of the estimated 
c                  bound for the spectral radius 
c                  (rounded to the nearest integer).
c     IWORK(12)   =Minimum value of the estimated  
c                  bound for the spectral radius 
c                  (rounded to the nearest integer).
c     
c   
c    Caution:     The variable UROUND (the rounding unit) is set to 
c    -------      1.0d-16 and may depends on the machines.
c-------------------------------------------------------------------
c *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** 
c         Numerical method
c *** *** *** *** *** *** *** *** *** *** *** *** *** *** ***
c------------------------------------------------------------------
c
c *** *** *** *** *** *** *** *** *** *** *** *** *** *** ***          
c             Declarations 
c *** *** *** *** *** *** *** *** *** *** *** *** *** *** ***
c
      double precision y(neqn),work(*),winc(*),
     & t,tend,h,uround
      integer iwork(15),neqn,i,n1,n2,n3,n4,n5,n6
     & nsto,idid,nsteps
      logical arret
      external f,g,genrnd
c -------- Uround: smallest number satisfying 1.d0+uround>1.d0
      data uround/1.0d-16/ 
c             
c *** *** *** *** *** *** *** *** *** *** *** *** *** *** ***
c             Data of the stability polynomials
c *** *** *** *** *** *** *** *** *** *** *** *** *** *** ***

c *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** 
c             Initializations
c *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** 
      arret=.false.
c -------- Prepare the entry-points for the arrays in work.--------     
      n1=1
      n2=n1+neqn
      n3=n2+neqn
      n4=n3+neqn
			n5=n4+neqn
			n6=n5+neqn
c -------- Call to the core integrator. -----------
      call rockcore(neqn,nsto,t,nsteps,h,winc,y,f,g,work,work(n1),
     & work(n2),work(n3),work(n4),work(n5),work(n6),
     & iwork,arret,uround,idid)
      return
      end
c ----------------------------------------------
c     End of subroutine SROCK1.
c ----------------------------------------------
c
      subroutine rockcore(neqn,nsto,t,nsteps,h,winc,y,f,g,work,
     & yn,fn,yjm1,yjm2,yjm3,yjm4,iwork,
     & arret,uround,idid)
c ----------------------------------------------
c    Core integrator for SROCK1.
c ---------------------------------------------- 
c             Declarations
c ----------------------------------------------
      implicit double precision (a-h,o-z)
       double precision y(*),yn(*),fn(*),work(*),sg(neqn),
     & yjm1(neqn),yjm2(neqn),yjm3(neqn),yjm4(neqn),yjm5(neqn),
     & tend,h,hnew,hp,facmax,fac,facp,rho,uround,yjm6(neqn),
     & eigmax,t,told,winc(*)
       integer iwork(15),neqn,nsto,mdeg,i,nsteps,istep,
     & nrho,mdego,nrej,idid
       logical last,reject,arret
       external f,g
			 double precision cks,tmp,rnd
			 integer j,k
c *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** 
c             Initializations
c *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** 
      iwork(5)=0
      iwork(6)=0
      iwork(7)=0
      iwork(8)=0
      iwork(9)=0
      iwork(10)=0      
      iwork(11)=0
      iwork(12)=0
      told=0.d0
      last=.false.
      reject=.false. 
      nrho=0
      idid=1
c			mdego=mdeg
			mdego=0
c -------- Big loop
      do istep=1,nsteps
				sqrth=sqrt(h)
				CALL GENRND(NSTO,WINC)
				DO I=1,NSTO
				WINC(I)=WINC(I)*sqrth
				END DO
c -------- Initialization of the integration step.--------   
10    do i=1,neqn
       yn(i)=y(i)
      end do

      if (iwork(4).lt.0) then
        write(6,*) 'ERROR: The number of stages must be positive!'
        stop
      end if
c -------- Spectral radius.--------		
      if (iwork(4).gt.0) then
c force the degree 
        mdeg=iwork(4)
				    eta=0.05d0
        call coeff(mdeg,eta,w01,w11,c,alpha)
c				if (mdeg.gt.200)  then
c           write (6,*) 'Warning: stages>200'
c           stop
c				end if
			else
c ------- Computed externally by rho.--------
          if (iwork(1).eq.1) then
            eigmax=rho(neqn,t,yn)
            if (idnint(eigmax).gt.iwork(11)) 
     &         iwork(11)=idnint(eigmax)
            if (iwork(6).eq.0) iwork(12)=iwork(11)
            if (idnint(eigmax).lt.iwork(12)) 
     &         iwork(12)=idnint(eigmax)
c ------- Computed internally by rocktrho.--------  
        else
				    call f(neqn,t,yn,fn)
						iwork(9)=iwork(9)+1
            call rocktrho(neqn,t,y,f,yn,fn,work,yjm1,
     &                   yjm2,eigmax,uround,idid,iwork)
            if (idnint(eigmax).gt.iwork(11)) 
     &         iwork(11)=idnint(eigmax)
            if (iwork(6).eq.0) iwork(12)=iwork(11)
            if (idnint(eigmax).lt.iwork(12)) 
     &         iwork(12)=idnint(eigmax)
          end if
        end if

c -------- The number of stages.--------
     			
c     compute optimal degree mdeg and eta for a given eigmax
      if(iwork(4).eq.0)  then
       call mdegr(h*eigmax,mdeg,eta)
       call coeff(mdeg,eta,w01,w11,c,alpha)
      end if
       

			
      if (mdeg.gt.iwork(10)) iwork(10)=mdeg
c -------- Computation of an integration step.--------
c SK-ROCK method
c Compute CHEBYCHEV METHOD OF ORDER ONE

c noise term		
			do i=1,neqn
				yjm2(i)=yn(i)
			end do		

			do i=1,neqn
			sg(i)=0.d0
			end do
			do j=1,nsto
			call g(neqn,nsto,j,t,yjm2,yjm3)
   
			do i=1,neqn
				sg(i)=sg(i)+mdeg*w11*yjm3(i)*winc(j)
			end do
			end do
      A0=1.0D0
      A1=W01
			cheb1mu=W11/A1
c Do first stage here
      
      
			   do i=1,neqn
				     yjm4(i)=yn(i)+0.5d0*sg(i)
         yjm5(i)=yn(i)-0.5d0*sg(i)
			   end do
      if (iwork(3).ne.1) then
			   call f(neqn,t,yjm4,fn)
			   iwork(5)=iwork(5)+1
			   DO I=1,NEQN	
			   yjm1(I)=Y(I)
			   yjm2(I)=Y(I)+H*cheb1mu*FN(I)+sg(I)/w01
      END DO
      else
 			  call f(neqn,t,yjm4,fn)
 			  iwork(5)=iwork(5)+1
			   call f(neqn,t,yjm5,yjm6)
			   iwork(5)=iwork(5)+1
      call f(neqn,t,yn,yjm3)
      iwork(5)=iwork(5)+1
      beta=2.0d0*alpha/(mdeg**2*w11**2)
			   DO I=1,NEQN	
			   yjm1(I)=Y(I)
			   yjm2(I)=Y(I)+((cheb1mu+beta)*h*fn(i)+beta*h*yjm6(i)
     & -2.d0*beta*h*yjm3(i))+sg(I)/w01
		    END DO
      end if
      
c----					
      
    			DO I=2,mdeg
          A2=2.0D0*W01*A1-A0
    			cheb1mu=2.0D0*W11*A1/A2
    			cheb1nu=2.0D0*W01*A1/A2
c			cheb1ka=A0/A2
          cheb1ka=1.d0-cheb1nu
          
			   call f(neqn,t,yjm2,yjm4)
      iwork(5)=iwork(5)+1    
			   DO J=1,NEQN
			   yjm3(J)=H*cheb1mu*yjm4(J)
     &   +cheb1nu*yjm2(J)+cheb1ka*yjm1(J)
c modification for SK-ROCK method
		     yjm1(J)=yjm2(J)
		     yjm2(J)=yjm3(J)
			   END DO
      A0=A1
      A1=A2
      END DO
c ---
			   DO I=1,NEQN
			   y(I)=yjm2(I)
			   END DO
      
c -------------------------------------------------------
      iwork(6)=iwork(6)+1  
      iwork(7)=iwork(7)+1
      t=t+h
c save solution in sol.dat
c			write (10,*) (Y(I),I=1,NEQN)
			

			   end do
      if (iwork(3).eq.1) then

   			CALL GENRND(NSTO,WINC)
   			DO I=1,NSTO
   			WINC(I)=WINC(I)*sqrth
   			END DO

      do j=1,nsto
			call g(neqn,nsto,j,t,yjm2,yjm5)
      DO I=1,NEQN
       y(I)=y(I)+c*yjm5(i)*winc(j)
       END DO
			end do
       end if
   			idid=1
      
      return      
      end
c ----------------------------------------------
c     End of subroutine rockcore.
c ----------------------------------------------  
c                

  

c   
      subroutine rocktrho(neqn,t,y,f,yn,fn,work,z,fz,eigmax,
     &                   uround,idid,iwork)
c------------------------------------------------------------ 
c     Rocktrho compute eigmax, a close upper bound of the
c     spectral radius of the Jacobian matrix using a 
c     power method (J.N. Franklin (matrix theory)). 
c     The algorithm used is a small change (initial vector
c     and stopping criteria) of that of
c     Sommeijer-Shampine-Verwer, implemented in RKC.
c-------------------------------------------------------------
c             Declarations
c-------------------------------------------------------------
       double precision y(neqn),yn(neqn),fn(neqn),z(neqn),
     & work(*),fz(neqn),t,eigmax,eigmaxo,sqrtu,uround,znor,
     & ynor,quot,dzyn,dfzfn,safe
       integer iwork(15),neqn,n5,i,iter,maxiter,nind,ntest,
     & ind,idid  
       parameter (maxiter=50)
       parameter (safe=1.2d0)
       external f
c *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** 
c             Initialisations
c *** *** *** *** *** *** *** *** *** *** *** *** *** *** ***     
      sqrtu=sqrt(uround)
      ynor=0.d0
      znor=0.d0
      n5=4*neqn
c ------ The initial vectors for the power method are yn --------
c       and yn+c*f(v_n), where vn=f(yn) a perturbation of yn 
c       (if iwork(6)=0) or a perturbation of the last computed
c       eigenvector (if iwork(6).neq.0). 
c
      if (iwork(6).eq.0) then
        do i=1,neqn
          fz(i)=fn(i)
        end do
        call f(neqn,t,fz,z)
        iwork(9)=iwork(9)+1
                         else
        do i=1,neqn
          z(i)=work(n5+i)
        end do
      end if
c ------ Perturbation.--------
      do i=1,neqn
        ynor=ynor+yn(i)**2
        znor=znor+z(i)**2
      end do
      ynor=sqrt(ynor)
      znor=sqrt(znor)
c ------ Normalization of the vector z so that --------
c        the difference z-yn lie in a circle 
c        around yen (ice has a constant modules).
c
      if (ynor.ne.0.d0.and.znor.ne.0.d0) then
        dzyn=ynor*sqrtu
        quot=dzyn/znor
        do i=1,neqn
          z(i)=yn(i)+z(i)*quot
        end do
      elseif(ynor.ne.0.d0) then
        dzyn=ynor*sqrtu
        do i=1,neqn
          z(i)=yn(i)+yn(i)*sqrtu
        end do
      elseif(znor.ne.0.d0) then
          dzyn=uround
          quot=dzyn/znor
          do i=1,neqn
            z(i)=z(i)*quot
          end do
      else
        dzyn=uround
        do i=1,neqn
          z(i)=dzyn
        end do
      end if
c ------ Start the power method.--------
      eigmax=0.d0
      do iter=1,maxiter
        call f(neqn,t,z,fz)
        iwork(9)=iwork(9)+1
        dfzfn=0.d0
        do i=1,neqn
          dfzfn=dfzfn+(fz(i)-fn(i))**2
        end do
        dfzfn=dsqrt(dfzfn)
        eigmaxo=eigmax
        eigmax=dfzfn/dzyn
        eigmax=safe*eigmax
c ------ The stopping criteria is based on a 
c        relative error between two successive
c        estimation ``eigmax'' of the spectral 
c        radius.
c 
        if (iter.ge.2.and.dabs(eigmax-eigmaxo)
     &    .le.(eigmax*0.05d0)) then
c ----- The last eigenvector is stored.--------
          do i=1,neqn
            work(n5+i)=z(i)-yn(i)
          end do
          return
        end if
c ----- The next z is defined by --------
c       z_new=yn+coef*(fz-fn) where
c       coef is chosen so that
c       norm(z_new-yn)=norm(z_old-yn).
c
        if (dfzfn.ne.0.d0) then
          quot=dzyn/dfzfn
          do i=1,neqn
            z(i)=yn(i)+(fz(i)-fn(i))*quot
          end do
        else
c ----- The new z is defined by an arbitrary --------
c       perturbation of the current approximation
c       of the eigenvector.
c
          nind=neqn
          ntest=0
          ind=1+mod(iter,nind)
          if (z(ind).ne.yn(ind).or.ntest.eq.10) then
            z(ind)=yn(ind)-(z(ind)-yn(ind))
          else 
            nind=neqn+ind
            ntest=ntest+1
          end if
        end if
      end do
      write(6,*) 'convergence failure in the 
     & spectral radius computation'
      idid=-3
      return
      end

      subroutine mdegr(rho,mdeg,eta)
c-------------------------------------------------------------          
c       Find the optimal degree and eta.
c-------------------------------------------------------------        
c ---------------------------------------------- 
c             Declarations
c ---------------------------------------------- 
      integer mdeg
			double precision eta,rho,c
			eta=0.05d0
c			c=2.d0-4.d0*eta/3.d0
      c=2.d0*tanh(sqrt(2.d0*eta))/sqrt(2.d0*eta)
c *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** 
c             Initialisations
c *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** 
			   mdeg=sqrt((rho+1.5d0)/c)+1.d0
c			   mdeg=nint(sqrt((rho+1.5d0)/c)+1.d0)
			   if (rho.lt.c) mdeg=1.d0
      return
      end  
c ----------------------------------------------
c     End of subroutine mdegr.
c ----------------------------------------------
    
    
c ----------------------------------------------
c     the Subroutine generating w1, w0, c, alpha
c ----------------------------------------------
       
      subroutine coeff(mdeg,eta,w01,w11,c,alpha)
      implicit double precision (a-h,o-z)
   			

      w01=1.0d0+eta/mdeg**2
      A0=1.0D0
      A1=W01
			   B0=0.0D0
      B1=1.0D0
      C0=0.d0
      C1=0.d0
      DO I=2,mdeg
      A2=2.0D0*W01*A1-A0
			   B2=2.0D0*W01*B1+2.0D0*A1-B0
      C2=2.0d0*w01*C1+4.0d0*B1-C0
      A0=A1
      A1=A2
			   B0=B1
      B1=B2
      C0=C1
      C1=C2
      END DO
			   W11=A1/B1
      c=sqrt(-1.d0/4.d0+w11/2.d0+C1*W11/B1-w11**2*C1/(4.d0*A1))
      alpha=(c**2+w11**2*C1/(2.d0*A1)-RS(mdeg,eta))*mdeg*w11/w01
      
      return
      end
         
c ------------------------------------------------
c     end of the Subroutine generating w1 and w0
c ------------------------------------------------      
c ----------------------------------------------
c     the Function RS 
c ----------------------------------------------
      double precision function RS(mdeg,eta)
      implicit double precision (a-h,o-z)
      
      w01=1.0d0+eta/mdeg**2
      
      A0=1.0D0
      A1=W01
			   B0=0.0D0
      B1=1.0D0
   			DO I=2,mdeg
         A2=2.0D0*W01*A1-A0
         B2=2.0D0*W01*B1+2.0D0*A1-B0
         A0=A1
         A1=A2
   			   B0=B1
         B1=B2
         END DO
         w11=A1/B1
         
         A0=1.0D0
         A1=W01
   			   B0=0.0D0
         B1=1.0D0
         r0=0.0d0
         r1=mdeg**2*w11**3/4.0d0*w01
         
         DO I=2,mdeg
          A2=2.0D0*W01*A1-A0
          B2=2.0D0*W01*B1+2.0D0*A1-B0
       			cheb1mu=2.0D0*W11*A1/A2
       			cheb1nu=2.0D0*W01*A1/A2
       			cheb1ka=A0/A2
          r2=cheb1nu*r1-cheb1ka*r0+cheb1mu*mdeg*B1*w11/((I-1)*A1)
          A0=A1
          A1=A2
          B0=B1
          B1=B2
          r0=r1
          r1=r2
          
          
         END DO
         
         RS=r1
         
      return
      end
     